package ru.yandex.stockpile.client.benchmarks;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.ColumnFlagMask;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@State(Scope.Benchmark)
public class ClusterWithData {
    private Cluster state;
    private List<MetricId> existsLines;

    @Param({"0", "1000", "5074"})
    private int countPoints;

    @Setup
    public void setUp(Cluster state) throws IOException, ExecutionException, InterruptedException {
        this.state = state;

        ThreadLocalRandom random = ThreadLocalRandom.current();

        CreateMetricRequest createRequest = CreateMetricRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .build();

        List<MetricId> ids = new ArrayList<>(1000);
        for (int index = 0; index < 1000; index++) {
            state.getClient().forceUpdateClusterMetaData().join();

            MetricId metricId = createMetric(state.getClient(), createRequest);
            ids.add(metricId);

            if (countPoints == 0) {
                continue;
            }

            long initTime = System.currentTimeMillis();
            List<TPoint> points = IntStream.range(0, countPoints)
                    .mapToObj(pointNum -> TPoint.newBuilder()
                            .setTimestampsMillis(pointNum * 15000 + initTime)
                            .setDoubleValue(random.nextDouble(0, 100000))
                            .build()
                    ).collect(Collectors.toList());

            syncWriteOne(state.getClient(), TWriteRequest.newBuilder()
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .setMetricId(metricId)
                    .addAllPoints(points)
                    .build()
            );
        }

        existsLines = ids;
    }

    @TearDown
    public void terminate() throws InterruptedException {
        state.terminate();
    }

    public MetricId nextMetric() {
        return existsLines.get(ThreadLocalRandom.current().nextInt(existsLines.size()));
    }

    public StockpileClient getClient() {
        return state.getClient();
    }

    private MetricId createMetric(StockpileClient client, CreateMetricRequest request) throws ExecutionException, InterruptedException {
        CreateMetricResponse response = client.createMetric(request).join();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ": " + response.getStatusMessage());
        }

        return response.getMetricId();
    }

    private void syncWriteOne(StockpileClient client, TWriteRequest request) throws ExecutionException, InterruptedException {
        TWriteResponse response = client.writeOne(request).get();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ": " + response.getStatusMessage());
        }
    }
}
