package ru.yandex.stockpile.client.benchmarks;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.stockpile.api.EColumnFlag;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.client.ColumnFlagMask;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@State(Scope.Benchmark)
@Threads(1)
@Fork(value = 1)
@Measurement(iterations = 5, time = 5, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 5, timeUnit = TimeUnit.SECONDS)
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class JmhClientWriteOneBenchmark {

    @Param({"1", "10", "100", "1000", "10000", "100000"})
    private int inflighRequests;

    @Param({"1", "100", "1000", "10000", "100000"})
    private int countPointInRequest;

    /*

Benchmark                                     (countNodes)  (countPointInRequest)  (countPoints)  (inflighRequests)  Mode  Cnt     Score     Error  Units
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0                  1  avgt    5     0.093 ±   0.035  ms/op
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0                 10  avgt    5     0.282 ±   0.023  ms/op
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0                100  avgt    5     2.383 ±   0.780  ms/op
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0               1000  avgt    5    21.582 ±   2.916  ms/op
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0              10000  avgt    5   215.520 ±  37.211  ms/op
JmhClientWriteOneBenchmark.writeManyAsStream             1                      1              0             100000  avgt    5  2206.901 ± 316.089  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0                  1  avgt    5     0.098 ±   0.024  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0                 10  avgt    5     0.229 ±   0.049  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0                100  avgt    5     0.467 ±   0.061  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0               1000  avgt    5     2.238 ±   0.497  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0              10000  avgt    5    20.091 ±   3.338  ms/op
JmhClientWriteOneBenchmark.writeOneBatched               1                      1              0             100000  avgt    5   197.222 ±  25.932  ms/op

     */

    public static void main(String[] args) throws RunnerException, IOException {
        Options opt = new OptionsBuilder()
                .include(JmhClientWriteOneBenchmark.class.getName())
                .detectJvmArgs()
                .jvmArgsAppend("-Xmx3g", "-Xms3g")
                .param("countNodes", "1")
                .param("inflighRequests", "1", "10", "100", "1000", "10000", "100000")
                .param("countPoints", "0")
                .param("countPointInRequest", "1")
                .build();

        new Runner(opt).run();
    }

    @Benchmark
    public Object writeOne(ClusterWithData cluster) {
        List<TWriteRequest> requests = prepareWriteRequests(cluster);

        StockpileClient client = cluster.getClient();
        CompletableFuture[] futures = requests.parallelStream()
                .map(client::writeOne)
                .toArray(CompletableFuture[]::new);

        return CompletableFuture.allOf(futures).join();
    }

    private List<TWriteRequest> prepareWriteRequests(ClusterWithData cluster) {
        List<TWriteRequest> requests = new ArrayList<>(inflighRequests);
        for (int index = 0; index < inflighRequests; index++) {
            MetricId metricId = cluster.nextMetric();
            List<TPoint> points = new ArrayList<>(countPointInRequest);
            for (int j = 0; j < countPointInRequest; j++) {
                points.add(
                        TPoint.newBuilder()
                                .setDoubleValue(ThreadLocalRandom.current().nextDouble())
                                .setTimestampsMillis(System.currentTimeMillis())
                                .build()
                );
            }

            TWriteRequest request = TWriteRequest.newBuilder()
                    .setMetricId(metricId)
                    .setColumnMask(ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP, EColumnFlag.COLUMN_DOUBLE))
                    .addAllPoints(points)
                    .build();

            requests.add(request);
        }
        return requests;
    }
}
