package ru.yandex.stockpile.client;

import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 * @author alexlovkov
 */
@Configuration
@ParametersAreNonnullByDefault
@Import({
        GrpcConfigurationContext.class
})
public class GrpcStockpileClientConfigContext implements DisposableBean {

    private final TStockpileClientConfig config;
    private final ThreadPoolProvider threadPoolProvider;

    @Nullable
    private StockpileClient client;

    @Autowired
    public GrpcStockpileClientConfigContext(TStockpileClientConfig config, ThreadPoolProvider threadPoolProvider) {
        this.config = config;
        this.threadPoolProvider = threadPoolProvider;
    }

    @Bean
    public StockpileClient grpcStockpileClient(
            @Qualifier("ClientId") Optional<String> clientId,
            ClientOptionsFactory clientOptionsFactory)
    {
        var opts = StockpileClientOptions.newBuilder(
                        clientOptionsFactory.newBuilder(
                        "StockpileClientConfig",
                        config.getGrpcConfig())
                    .setClientId(
                        clientId.map(Strings::emptyToNull).orElseGet(HostUtils::getShortName)))
            .setFromConfig(config)
            .setRetryStopStrategy(StopStrategies.stopAfterAttempt(3))
            .build();

        var addresses = config.getGrpcConfig().getAddressesList();
        client = StockpileClients.createDynamic(addresses, opts);
        return client;
    }

    @Override
    public void destroy() {
        StockpileClient client = this.client;
        if (client != null) {
            client.close();
        }
    }
}
