package ru.yandex.stockpile.client;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ParametersAreNonnullByDefault
public final class MetricCreatePolicy {
    private static final MetricCreatePolicy DEFAULT_POLICY = MetricCreatePolicy.newBuilder().build();
    private final double percentOfReadyShards;
    private final double choosePercent;
    private final ShardCriteria criteria;

    private MetricCreatePolicy(Builder builder) {
        this.percentOfReadyShards = builder.percentOfReadyShards;
        this.choosePercent = builder.choosePercent;
        this.criteria = builder.criteria;
    }

    /**
     * <ul>
     * <li>
     * Enable create only if 60% of shards ready
     * </li>
     * <li>
     * Use record count on shard as criteria for choose
     * </li>
     * <li>
     * Chose shard randomly from 50% of shard sorted by record count
     * </li>
     * </ul>
     */
    @Nonnull
    public static MetricCreatePolicy defaultPolicy() {
        return DEFAULT_POLICY;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public double getPercentOfReadyShards() {
        return percentOfReadyShards;
    }

    public double getChoosePercent() {
        return choosePercent;
    }

    @Nonnull
    public ShardCriteria getCriteria() {
        return criteria;
    }

    @Override
    public String toString() {
        return "SensorCreatePolicy{" +
                "percentOfReadyShards=" + percentOfReadyShards +
                ", choosePercent=" + choosePercent +
                ", criteria=" + criteria +
                '}';
    }

    public enum ShardCriteria {
        /**
         * Count metrics in shard
         */
        METRIC_COUNT,

        /**
         * Sum records over all metrics created in shard
         */
        RECORD_COUNT,
        ;
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private Double percentOfReadyShards = 0.6;
        private Double choosePercent = 0.5;
        private ShardCriteria criteria = ShardCriteria.RECORD_COUNT;

        /**
         * Enable create new metrics only when percent of ready shard will be great or equal specified parameter.
         * By default parameter equal to 60%.
         * @param percent 0 < value <= 1
         */
        public Builder createWhenPercentOfReadyShard(double percent) {
            if (percent <= 0d || percent > 1d) {
                throw new IllegalArgumentException("Percent should be between greater then 0 and less then 1, but specified: " + percent);
            }
            this.percentOfReadyShards = percent;
            return this;
        }

        /**
         * Ready shards sorts by chose criteria and after that from this list randomly select one shard that
         * will be use to create metric. Random select shard necessary because metadata refresh with some delay and
         * it's means if we will select always first shard from sorted list multiple client can create multiple metrics
         * on this shard and as a result shard will be overloaded.
         *
         * By default use 50% of ready and sorted shards to select shard for metric create.
         *
         * @param percent 0 < value <= 1
         */
        @Nonnull
        public Builder chooseShardFromPercent(double percent) {
            if (percent <= 0d || percent > 1d) {
                throw new IllegalArgumentException("Percent should be between greater then 0 and less then 1, but specified: " + percent);
            }
            this.choosePercent = percent;
            return this;
        }

        /**
         * Criteria by that will be sorted shards.
         */
        @Nonnull
        public Builder chooseByCriteria(ShardCriteria criteria) {
            this.criteria = criteria;
            return this;
        }

        @Nonnull
        public MetricCreatePolicy build() {
            return new MetricCreatePolicy(this);
        }
    }
}
