package ru.yandex.stockpile.client;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.collect.Range;

import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.ReadMetricsMetaResponse;
import ru.yandex.stockpile.api.TAllocateLocalIdsRequest;
import ru.yandex.stockpile.api.TAllocateLocalIdsResponse;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest;
import ru.yandex.stockpile.api.TDeleteMetricByShardResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.api.TWriteDataBinaryResponse;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteLogResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.mem.AccumulatedShardCommand;

/**
 * @author Vladimir Gordiychuk
 */
@ThreadSafe
@ParametersAreNonnullByDefault
public interface StockpileClient extends AutoCloseable {

    /**
     * <p>
     * Create metric in particular shard.
     * <p>
     * If shardId was not provided, then system will automatically choose
     * least loaded shard in cluster.
     * <p>
     * If localId was not provided, then system will automatically generate a unique id for the metric.
     * <p>
     * <b>Recommendations:</b>
     * <ul>
     * <li>Use the same shardId for data that requested together to avoid cross shard request.</li>
     * <li>Don't put many data on the same shard because it lead to shard overload and
     * as a result performance degradation on write/read operations.</li>
     * </ul>
     * <p>
     */
    CompletableFuture<CreateMetricResponse> createMetric(CreateMetricRequest request);

    /**
     * Delete metric and all its associated data from stockpile. Physical process of metric
     * removing from storage can took up to couple days and that is why it's not recommended
     * to reuse {@link ru.yandex.solomon.model.protobuf.MetricId} right after metric deletion.
     */
    CompletableFuture<DeleteMetricResponse> deleteMetric(DeleteMetricRequest request);

    /**
     * Delete a range of data from metric. Used resource for store this
     * data can be free later, because it's required few days, but for read deleted data
     * should not be visible.
     */
    CompletableFuture<DeleteMetricDataResponse> deleteMetricData(DeleteMetricDataRequest request);

    /**
     * Write points into one metric. If metric was not create yet,
     * stockpile return correspond error.
     */
    CompletableFuture<TWriteResponse> writeOne(TWriteRequest request);

    /**
     * Write to already created metric compressed data.
     */
    CompletableFuture<TCompressedWriteResponse> writeCompressedOne(TCompressedWriteRequest request);

    CompletableFuture<TWriteDataBinaryResponse> writeDataBinary(TWriteDataBinaryRequest request);

    CompletableFuture<TWriteLogResponse> writeLog(TWriteLogRequest request);

    /**
     * Atomic apply multiple commands under the same shard
     */
    CompletableFuture<TShardCommandResponse> bulkShardCommand(TShardCommandRequest request);

    /**
     * Atomic apply multiple commands under the same shard
     */
    CompletableFuture<TShardCommandResponse> bulkShardCommand(@WillClose AccumulatedShardCommand commands);

    /**
     * Read points from one metric. If metric was not create yet,
     * stockpile return correspond error.
     */
    CompletableFuture<TReadResponse> readOne(TReadRequest readRequest);

    CompletableFuture<TUncompressedReadManyResponse> readUncompressedMany(TReadManyRequest request);

    /**
     * Read from exists metric compressed time series.
     * Read request should contain compression protocol version.
     */
    CompletableFuture<TCompressedReadResponse> readCompressedOne(TReadRequest request);

    CompletableFuture<TCompressedReadManyResponse> readCompressedMany(TReadManyRequest request);

    CompletableFuture<ReadMetricsMetaResponse> readMetricsMeta(ReadMetricsMetaRequest request);

    /**
     * @return range with supported compression format by the whole cluster, empty range can indicate that
     * cluster not initialized yet or range supported version on particular node not contains intersection with other.
     */
    Range<Integer> getCompatibleCompressFormat();

    /**
     * Collect information about cluster state(where located particular shard,
     * per shard load, etc). Client automatically refresh metadata with configured interval.
     * Force refresh metadata can be necessary after rebalance cluster,
     * or when response have one of the next status:
     * {@link EStockpileStatusCode#SHARD_ABSENT_ON_HOST}/{@link EStockpileStatusCode#SHARD_NOT_READY}.
     */
    CompletableFuture<Void> forceUpdateClusterMetaData();

    CompletableFuture<TAllocateLocalIdsResponse> allocateLocalIds(TAllocateLocalIdsRequest request);

    CompletableFuture<TDeleteMetricByShardResponse> deleteMetricByShard(TDeleteMetricByShardRequest request);

    /**
     * Count of shards ready to accept requests
     */
    int getReadyShardsCount();

    /**
     * Total amount of shards in a system
     */
    int getTotalShardsCount();

    /**
     * @return from 0 to 1 where 1 means that everything ready to accept requests
     */
    AvailabilityStatus getAvailability();

    String getHostForShardId(int shardId);

    /**
     * Terminate all active request and close connections
     */
    @Override
    void close();
}
