package ru.yandex.stockpile.client;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Iterables;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.FilterDiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileClientFactory {
    private final ThreadPoolProvider threadPoolProvider;
    private final ClusterMapper clusterMapper;
    private final MetricRegistry registry;
    private final ClientOptionsFactory clientOptionsFactory;

    public StockpileClientFactory(
            ThreadPoolProvider threadPoolProvider,
            ClusterMapper clusterMapper,
            MetricRegistry registry,
            ClientOptionsFactory clientOptionsFactory)
    {
        this.threadPoolProvider = threadPoolProvider;
        this.clusterMapper = clusterMapper;
        this.registry = registry;
        this.clientOptionsFactory = clientOptionsFactory;
    }

    public StockpileClient createClient(String clientId, TStockpileClientConfig config) {
        return Iterables.getOnlyElement(createClients(clientId, config).entrySet()).getValue();
    }

    public Map<String, StockpileClient> createClients(String clientId, TStockpileClientConfig config) {
        Map<String, StockpileClient> result = new HashMap<>();
        for (String clusterId : clusterMapper.knownClusterIds()) {
            var discovery = new FilterDiscoveryService(DiscoveryService.async(), (address) -> {
                return Objects.equals(clusterId, clusterMapper.byFqdnOrNull(address.getHost()));
            });
            var options = StockpileClientOptions.newBuilder(
                    clientOptionsFactory.newBuilder(
                            "StockpileClientConfig",
                            config.getGrpcConfig())
                        .setClientId(clientId)
                        .setMetricRegistry(registry.subRegistry("target", clusterId)))
                .setFromConfig(config)
                .setDiscoveryService(discovery)
                .build();

            StockpileClient client = StockpileClients.createDynamic(config.getGrpcConfig().getAddressesList(), options);
            result.put(clusterId, client);
        }
        return result;
    }
}
