package ru.yandex.stockpile.client;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;

import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ParametersAreNonnullByDefault
public final class StockpileClientOptions {
    private static final long DEFAULT_CLUSTER_METADATA_EXPIRE_MILLIS = TimeUnit.MINUTES.toMillis(1);
    private static final long DEFAULT_CLUSTER_METADATA_RETRY_DELAY_MILLIS = 500L;

    private static final StockpileClientOptions EMPTY = newBuilder(DefaultClientOptions.empty()).build();

    private final GrpcClientOptions grpcOptions;

    private final long metadataExpireMs;
    private final long metadataRequestTimeoutMs;
    private final MetricCreatePolicy metricCreatePolicy;
    private final long metadataRetryDelayMs;
    private final StopStrategy retryStopStrategy;
    private final DiscoveryService discoveryService;

    private StockpileClientOptions(Builder builder) {
        this.grpcOptions = builder.grpcOptions;

        this.metadataExpireMs = Optional.ofNullable(builder.metadataExpireMs)
                .orElse(DEFAULT_CLUSTER_METADATA_EXPIRE_MILLIS);
        this.metadataRequestTimeoutMs = Optional.ofNullable(builder.metadataRequestTimeoutMs)
                .orElseGet(
                    () -> grpcOptions.getDefaultTimeoutMillis() > 0
                        ? grpcOptions.getDefaultTimeoutMillis()
                        : DEFAULT_CLUSTER_METADATA_EXPIRE_MILLIS);

        this.metricCreatePolicy = Optional.ofNullable(builder.metricCreatePolicy)
                .orElseGet(MetricCreatePolicy::defaultPolicy);

        this.metadataRetryDelayMs = Optional.ofNullable(builder.metadataRetryDelayMs)
                .orElse(DEFAULT_CLUSTER_METADATA_RETRY_DELAY_MILLIS);
        this.retryStopStrategy = Optional.ofNullable(builder.retryStopStategy)
                .orElseGet(StopStrategies::alwaysStop);
        this.discoveryService = Optional.ofNullable(builder.discoveryService).orElseGet(DiscoveryService::async);
    }

    public static Builder newBuilder(GrpcClientOptions.Builder grpcOptionsBuilder) {
        return newBuilder(grpcOptionsBuilder.build());
    }

    public static Builder newBuilder(GrpcClientOptions grpcOptions) {
        return new Builder(grpcOptions);
    }

    public static StockpileClientOptions empty() {
        return EMPTY;
    }

    public GrpcClientOptions getGrpcOptions() {
        return grpcOptions;
    }

    public long getMetadataExpireMs() {
        return metadataExpireMs;
    }

    public long getMetadataRequestTimeoutMs() {
        return metadataRequestTimeoutMs;
    }

    public MetricCreatePolicy getMetricCreatePolicy() {
        return metricCreatePolicy;
    }

    public long getMetadataRetryDelayMs() {
        return metadataRetryDelayMs;
    }

    public StopStrategy getRetryStopStrategy() {
        return retryStopStrategy;
    }

    public DiscoveryService getDiscoveryService() {
        return discoveryService;
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {

        private final GrpcClientOptions grpcOptions;

        @Nullable
        private Long metadataExpireMs;
        @Nullable
        private Long metadataRequestTimeoutMs;
        @Nullable
        private MetricCreatePolicy metricCreatePolicy;
        @Nullable
        private Long metadataRetryDelayMs;
        @Nullable
        private StopStrategy retryStopStategy;
        @Nullable
        private DiscoveryService discoveryService;

        public Builder(GrpcClientOptions grpcOptions) {
            this.grpcOptions = grpcOptions;
        }

        /**
         * Define how much time metadata about cluster state will be relevant.
         * Client will be automatically refresh metadata with specified delay.
         * <p>
         * Too high delay can lead avoid d to errors with code
         * {@link ru.yandex.stockpile.api.EStockpileStatusCode#SHARD_ABSENT_ON_HOST}
         * or {@link ru.yandex.stockpile.api.EStockpileStatusCode#SHARD_NOT_READY},
         * but too low delay lead to many requests to server.
         */
        public Builder setExpireClusterMetadata(long value, TimeUnit unit) {
            this.metadataExpireMs = unit.toMillis(value);
            return this;
        }

        /**
         * Timeout for metadata requests. These parameters also define max latency between background
         * metadata refresh. It means that next metadata refresh occurs only when collected answer
         * from each node, and if one of the node failed and not answer background process will wait
         * time out.
         * <p>
         * It's an optional parameter, If the user has not provided timeout when the client is build,
         * client will use default request timeout ({@link GrpcClientOptions#getDefaultTimeoutMillis()})
         */
        public Builder setMetaDataRequestTimeOut(long value, TimeUnit unit) {
            this.metadataRequestTimeoutMs = unit.toMillis(value);
            return this;
        }

        /**
         * Define policy that will be apply for each metric create. If parameter not specified
         * will be use default policy({@link MetricCreatePolicy#defaultPolicy()})
         *
         * @see MetricCreatePolicy
         */
        public Builder setCreateMetricPolicy(MetricCreatePolicy metricCreatePolicy) {
            this.metricCreatePolicy = metricCreatePolicy;
            return this;
        }

        /**
         * Background update metadata can fail on particular node. This parameter define retry delay that
         * increments exponential for sequential fails on particular node. If parameter not specified
         * will be use default delay equal to 500ms.
         */
        public Builder setMetaDataRequestRetryDelay(long time, TimeUnit unit) {
            this.metadataRetryDelayMs = unit.toMillis(time);
            return this;
        }

        /**
         * Strategy that decide how long failed request will be retry. It's useful during network lags
         * when one of the node not available retry request after some time automatically.
         * <p>
         * It's optional parameter. If the user has not provided an strategy when the client options is build, will
         * be used default strategy that reject retries.
         */
        public Builder setRetryStopStrategy(StopStrategy stopStrategy) {
            this.retryStopStategy = stopStrategy;
            return this;
        }

        public Builder setDiscoveryService(@Nullable DiscoveryService discoveryService) {
            this.discoveryService = discoveryService;
            return this;
        }

        public Builder setFromConfig(TStockpileClientConfig config) {
            setTime(this::setExpireClusterMetadata, config.getMetadataExpireTime());
            setTime(this::setMetaDataRequestTimeOut, config.getMetadataRequestTimeout());
            setTime(this::setMetaDataRequestRetryDelay, config.getMetadataRetryDelayTimeout());
            return this;
        }

        public StockpileClientOptions build() {
            return new StockpileClientOptions(this);
        }
    }
}
