package ru.yandex.stockpile.client.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;
import javax.annotation.WillNotClose;

import io.grpc.Drainable;
import io.grpc.KnownLength;
import io.grpc.MethodDescriptor;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufInputStream;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ByteBufMarshaller implements MethodDescriptor.Marshaller<ByteBuf> {
    private static final InputStream CLOSED_INPUT_STREAM = new ClosedInputStream();

    // TODO ByteBuf leak during exception on gRPC call https://github.com/grpc/grpc-java/issues/3318
    @Override
    public InputStream stream(@WillNotClose ByteBuf value) {
        ByteBuf buffer = value.retain();
        return new AutoClosableInputStream(buffer);
    }

    @Override
    @WillNotClose
    public ByteBuf parse(InputStream stream) {
        ByteBuf buffer = null;
        try {
            if (stream instanceof KnownLength) {
                int size = stream.available();
                buffer = ByteBufAllocator.DEFAULT.buffer(size, size);
                buffer.writeBytes(stream, size);
                return buffer;
            } else {
                throw new UnsupportedOperationException("Not implemented yet!");
            }
        } catch (IOException e) {
            if (buffer != null) {
                buffer.release();
            }

            throw new RuntimeException(e);
        }
    }

    // TODO: WA to avoid memory leaks cause by gRPC. Drop it after fix https://github.com/grpc/grpc-java/issues/3318 (gordiychuk@)
    private static class AutoClosableInputStream extends InputStream implements KnownLength, Drainable {
        private ByteBuf buf;
        @WillClose
        private InputStream input;
        private boolean closed = false;

        private AutoClosableInputStream(@WillClose ByteBuf buf) {
            this.buf = buf;
            input = new ByteBufInputStream(buf, true);
        }

        @Override
        public int read() throws IOException {
            try {
                return input.read();
            } finally {
                closeIfNecessary();
            }
        }

        @Override
        public int read(byte[] b) throws IOException {
            try {
                return input.read(b);
            } finally {
                closeIfNecessary();
            }
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            try {
                return input.read(b, off, len);
            } finally {
                closeIfNecessary();
            }
        }

        @Override
        public long skip(long n) throws IOException {
            try {
                return input.skip(n);
            } finally {
                closeIfNecessary();
            }
        }

        @Override
        public int available() throws IOException {
            return input.available();
        }

        @Override
        public void close() throws IOException {
            if (closed) {
                return;
            }

            input.close();
            input = CLOSED_INPUT_STREAM;
            closed = true;
        }

        private void closeIfNecessary() {
            try {
                if (input.available() == 0) {
                    close();
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public int drainTo(OutputStream target) throws IOException {
            int size = buf.readableBytes();
            buf.readBytes(target, size);
            return size;
        }
    }

    private static class ClosedInputStream extends InputStream {
        private ClosedInputStream() {
        }

        @Override
        public int read() {
            return -1;
        }

        @Override
        public int read(byte[] b) {
            return -1;
        }

        @Override
        public int read(byte[] b, int off, int len) {
            // Event if requested zero bytes return -1 to avoid stuck https://github.com/grpc/grpc-java/issues/3323
            return -1;
        }
    }
}
