package ru.yandex.stockpile.client.impl;

import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.ToIntFunction;
import java.util.function.ToLongFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.MethodDescriptor;

import ru.yandex.stockpile.api.EStockpileStatusCode;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class EndpointDescriptor<ReqT, RespT> {
    private final MethodDescriptor<ReqT, RespT> method;
    private final ToIntFunction<ReqT> shardIdFn;
    private final ToLongFunction<ReqT> deadlineFn;
    private final Function<RespT, EStockpileStatusCode> responseStatusFn;
    private final BiFunction<EStockpileStatusCode, String, RespT> errorHandling;
    private final Predicate<Shard> readyPredicate;

    private EndpointDescriptor(Builder<ReqT, RespT> builder) {
        this.method = requireNonNull(builder.method);
        this.shardIdFn = requireNonNull(builder.shardIdFn);
        this.deadlineFn = requireNonNull(builder.deadlineFn);
        this.responseStatusFn = requireNonNull(builder.responseStatusFn);
        this.errorHandling = requireNonNull(builder.errorHandling);
        this.readyPredicate = requireNonNull(builder.readyPredicate);
    }

    public static <ReqT, RespT> Builder<ReqT, RespT> newBuilder() {
        return new Builder<>();
    }

    public MethodDescriptor<ReqT, RespT> getMethod() {
        return method;
    }

    public int getShardId(ReqT request) {
        return shardIdFn.applyAsInt(request);
    }

    public long getDeadline(ReqT request) {
        return deadlineFn.applyAsLong(request);
    }

    public EStockpileStatusCode getStatusCode(RespT response) {
        return responseStatusFn.apply(response);
    }

    public RespT handleError(EStockpileStatusCode code, String message) {
        return errorHandling.apply(code, message);
    }

    public boolean isReady(Shard shard) {
        return readyPredicate.test(shard);
    }

    public static class Builder<ReqT, RespT> {
        private MethodDescriptor<ReqT, RespT> method;
        private ToIntFunction<ReqT> shardIdFn;
        private ToLongFunction<ReqT> deadlineFn;
        private Function<RespT, EStockpileStatusCode> responseStatusFn;
        private BiFunction<EStockpileStatusCode, String, RespT> errorHandling;
        private Predicate<Shard> readyPredicate = Shard::isReady;

        public Builder() {
        }

        public Builder<ReqT, RespT> setGrpcMethod(MethodDescriptor<ReqT, RespT> method) {
            this.method = method;
            return this;
        }

        public Builder<ReqT, RespT> setShardIdResolver(ToIntFunction<ReqT> shardIdFn) {
            this.shardIdFn = shardIdFn;
            return this;
        }

        public Builder<ReqT, RespT> setDeadlineResolver(ToLongFunction<ReqT> deadlineFn) {
            this.deadlineFn = deadlineFn;
            return this;
        }

        public Builder<ReqT, RespT> setStatusCodeResolver(Function<RespT, EStockpileStatusCode> responseStatusFn) {
            this.responseStatusFn = responseStatusFn;
            return this;
        }

        public Builder<ReqT, RespT> setErrorHandling(BiFunction<EStockpileStatusCode, String, RespT> errorHandling) {
            this.errorHandling = errorHandling;
            return this;
        }

        public Builder<ReqT, RespT> setReadyPredicate(Predicate<Shard> predicate) {
            this.readyPredicate = predicate;
            return this;
        }

        public EndpointDescriptor<ReqT, RespT> build() {
            return new EndpointDescriptor<>(this);
        }
    }
}
