package ru.yandex.stockpile.client.impl;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Range;
import io.netty.util.concurrent.DefaultThreadFactory;

import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.ReadMetricsMetaResponse;
import ru.yandex.stockpile.api.TAllocateLocalIdsRequest;
import ru.yandex.stockpile.api.TAllocateLocalIdsResponse;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest;
import ru.yandex.stockpile.api.TDeleteMetricByShardResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.api.TWriteDataBinaryResponse;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteLogResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.Attempt;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StopStrategies;
import ru.yandex.stockpile.client.StopStrategy;
import ru.yandex.stockpile.client.mem.AccumulatedShardCommand;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_ALLOCATE_LOCAL_IDS;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_BULK_SHARD_COMMAND;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_BULK_SHARD_COMMAND_ACCUMULATED;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_CREATE_METRIC;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_DELETE_METRIC;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_DELETE_METRIC_BY_SHARD;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_DELETE_METRIC_DATA;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_READ_COMPRESSED_MANY;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_READ_COMPRESSED_ONE;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_READ_METRICS_META;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_READ_ONE;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_READ_UNCOMPRESSED_MANY;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_WRITE_COMPRESSED_ONE;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_WRITE_DATA_BINARY;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_WRITE_LOG;
import static ru.yandex.stockpile.client.impl.EndpointDescriptors.METHOD_WRITE_ONE;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class GrpcStockpileClient implements StockpileClient {
    private final Cluster cluster;
    private final ExecutorService responseHandlerExecutorService;
    private final boolean ownResponseHandlerExecutorService;
    private final StopStrategy retryStopStrategy;
    private final long defaultRequestTimeoutMs;

    public GrpcStockpileClient(List<String> addresses, StockpileClientOptions options) {
        this.cluster = new Cluster(addresses, options);
        this.cluster.forceClusterStatusUpdate();
        this.retryStopStrategy = options.getRetryStopStrategy();
        this.defaultRequestTimeoutMs = options.getGrpcOptions().getDefaultTimeoutMillis();

        Optional<ExecutorService> responseHandler = options.getGrpcOptions().getResponseHandlerExecutorService();
        if (responseHandler.isPresent()) {
            this.responseHandlerExecutorService = responseHandler.get();
            this.ownResponseHandlerExecutorService = false;
        } else {
            this.ownResponseHandlerExecutorService = true;
            this.responseHandlerExecutorService = Executors.newSingleThreadExecutor(
                    new DefaultThreadFactory("stockpile-client-response-handler-pool")
            );
        }
    }

    @Override
    public CompletableFuture<CreateMetricResponse> createMetric(CreateMetricRequest request) {
        if (request.getMetricId().getShardId() == 0) {
            try {
                request = request.toBuilder()
                        .setMetricId(request.getMetricId()
                                .toBuilder()
                                .setShardId(StockpileShardId.random(cluster.getTotalShardsCount())))
                        .build();
            } catch (NoSuitableMetricFound e) {
                return CompletableFuture.completedFuture(
                        CreateMetricResponse.newBuilder()
                                .setStatus(EStockpileStatusCode.NOTE_ENOUGH_READY_SHARDS)
                                .setStatusMessage(e.getMessage())
                                .build());
            }
        }

        return runWithRetry(METHOD_CREATE_METRIC, request);
    }

    @Override
    public CompletableFuture<DeleteMetricResponse> deleteMetric(DeleteMetricRequest request) {
        return runWithRetry(METHOD_DELETE_METRIC, request);
    }

    @Override
    public CompletableFuture<DeleteMetricDataResponse> deleteMetricData(DeleteMetricDataRequest request) {
        return runWithRetry(METHOD_DELETE_METRIC_DATA, request);
    }

    @Override
    public CompletableFuture<TWriteResponse> writeOne(TWriteRequest request) {
        return runWithRetry(METHOD_WRITE_ONE, request);
    }

    @Override
    public CompletableFuture<TCompressedWriteResponse> writeCompressedOne(TCompressedWriteRequest request) {
        return runWithRetry(METHOD_WRITE_COMPRESSED_ONE, request);
    }

    @Override
    public CompletableFuture<TWriteDataBinaryResponse> writeDataBinary(TWriteDataBinaryRequest request) {
        return run(METHOD_WRITE_DATA_BINARY, request);
    }

    @Override
    public CompletableFuture<TWriteLogResponse> writeLog(TWriteLogRequest request) {
        return run(METHOD_WRITE_LOG, request);
    }

    @Override
    public CompletableFuture<TShardCommandResponse> bulkShardCommand(TShardCommandRequest request) {
        return runWithRetry(METHOD_BULK_SHARD_COMMAND, request);
    }

    @Override
    public CompletableFuture<TShardCommandResponse> bulkShardCommand(AccumulatedShardCommand commands) {
        return runWithRetry(METHOD_BULK_SHARD_COMMAND_ACCUMULATED, commands)
                .whenComplete((response, e) -> commands.close());
    }

    @Override
    public CompletableFuture<TReadResponse> readOne(TReadRequest request) {
        return runWithRetry(METHOD_READ_ONE, request);
    }

    @Override
    public CompletableFuture<TUncompressedReadManyResponse> readUncompressedMany(TReadManyRequest request) {
        return runWithRetry(METHOD_READ_UNCOMPRESSED_MANY, request);
    }

    @Override
    public CompletableFuture<TCompressedReadResponse> readCompressedOne(TReadRequest request) {
        return runWithRetry(METHOD_READ_COMPRESSED_ONE, request);
    }

    @Override
    public CompletableFuture<TCompressedReadManyResponse> readCompressedMany(TReadManyRequest request) {
        return runWithRetry(METHOD_READ_COMPRESSED_MANY, request);
    }

    @Override
    public CompletableFuture<ReadMetricsMetaResponse> readMetricsMeta(ReadMetricsMetaRequest request) {
        return runWithRetry(METHOD_READ_METRICS_META, request);
    }

    @Override
    public Range<Integer> getCompatibleCompressFormat() {
        return cluster.getCompatibleCompressFormat();
    }

    @Override
    public CompletableFuture<Void> forceUpdateClusterMetaData() {
        return cluster.forceClusterStatusUpdate();
    }

    @Override
    public CompletableFuture<TAllocateLocalIdsResponse> allocateLocalIds(TAllocateLocalIdsRequest request) {
        return runWithRetry(METHOD_ALLOCATE_LOCAL_IDS, request);
    }

    @Override
    public CompletableFuture<TDeleteMetricByShardResponse> deleteMetricByShard(TDeleteMetricByShardRequest request) {
        return runWithRetry(METHOD_DELETE_METRIC_BY_SHARD, request);
    }

    @Override
    public int getReadyShardsCount() {
        return cluster.getReadyShardsCount();
    }

    public int getTotalShardsCount() {
        return cluster.getTotalShardsCount();
    }

    @Override
    public AvailabilityStatus getAvailability() {
        return cluster.getAvailability();
    }

    @Override
    public String getHostForShardId(int shardId) {
        Shard shard = cluster.getShard(shardId);
        if (shard == null) {
            throw new StockpileRuntimeException(EStockpileStatusCode.SHARD_NOT_READY, "shard location is unknown: " + shardId);
        }
        return shard.getFqdn();
    }

    @Override
    public void close() {
        if (ownResponseHandlerExecutorService) {
            responseHandlerExecutorService.shutdown();
        }
        this.cluster.close();
    }

    @Override
    public String toString() {
        return "GrpcStockpileClient{" + cluster + '}';
    }

    private long getDefaultDeadline() {
        if (defaultRequestTimeoutMs == 0) {
            return 0;
        }

        return System.currentTimeMillis() + defaultRequestTimeoutMs;
    }

    private <ReqT, RespT> CompletableFuture<RespT> run(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request) {
        int shardId = endpoint.getShardId(request);
        Shard shard = cluster.getShard(shardId);
        if (shard == null || !endpoint.isReady(shard)) {
            RespT resp = endpoint.handleError(EStockpileStatusCode.SHARD_NOT_READY, "Selected shard " + shardId + " is not ready yet");
            return CompletableFuture.completedFuture(resp);
        }

        return cluster.getClient(shard.getFqdn()).unaryCall(endpoint, request);
    }

    private <ReqT, RespT> CompletableFuture<RespT> runWithRetry(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request) {
        if (retryStopStrategy == StopStrategies.alwaysStop()) {
            return run(endpoint, request);
        }

        long deadline = endpoint.getDeadline(request) != 0
                ? endpoint.getDeadline(request)
                : getDefaultDeadline();

        CompletableFuture<RespT> future = new CompletableFuture<>();
        run(endpoint, request).whenCompleteAsync((response, e) -> {
            if (e != null) {
                future.completeExceptionally(e);
                return;
            }

            EStockpileStatusCode statusCode = endpoint.getStatusCode(response);
            if (!isRetryStatus(statusCode)) {
                future.complete(response);
                return;
            }

            Attempt<ReqT, RespT> attempt = new Attempt<>(1, request, response, deadline);
            retry(endpoint, attempt, future);
        }, responseHandlerExecutorService);
        return future;
    }

    private <ReqT, RespT> void retry(EndpointDescriptor<ReqT, RespT> endpoint, Attempt<ReqT, RespT> attempt, CompletableFuture<RespT> future) {
        if (retryStopStrategy.shouldStop(attempt)) {
            future.complete(attempt.getResponse());
            return;
        }

        final long deadline = attempt.getDeadline();
        if (deadline != 0 && deadline < System.currentTimeMillis()) {
            String message = String.format(
                    "Deadline %s exceeded during retry on %s attempt where latest response was %s",
                    Instant.ofEpochMilli(deadline),
                    attempt.getAttemptNumber(),
                    attempt.getResponse()
            );

            RespT response = endpoint.handleError(EStockpileStatusCode.DEADLINE_EXCEEDED, message);
            future.complete(response);
        }

        int shardId = endpoint.getShardId(attempt.getRequest());
        EStockpileStatusCode code = endpoint.getStatusCode(attempt.getResponse());
        updateClusterMataData(shardId, code)
                .whenCompleteAsync((ignore1, ignore2) -> {
                    run(endpoint, attempt.getRequest())
                            .whenCompleteAsync((response, e) -> {
                                if (e != null) {
                                    future.completeExceptionally(e);
                                    return;
                                }

                                EStockpileStatusCode statusCode = endpoint.getStatusCode(response);
                                if (!isRetryStatus(statusCode)) {
                                    future.complete(response);
                                    return;
                                }

                                retry(endpoint, attempt.next(response), future);
                            }, responseHandlerExecutorService);

                }, responseHandlerExecutorService);
    }

    private CompletableFuture<?> updateClusterMataData(int shardId, EStockpileStatusCode code) {
        return cluster.shardError(shardId, code);
    }

    private boolean isRetryStatus(EStockpileStatusCode code) {
        switch (code) {
            case SHARD_ABSENT_ON_HOST:
            case SHARD_NOT_READY:
            case NODE_UNAVAILABLE:
            case NOTE_ENOUGH_READY_SHARDS:
                return true;
            default:
                return false;
        }
    }
}
