package ru.yandex.stockpile.client.impl;

import java.nio.channels.ClosedChannelException;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import com.google.common.net.HostAndPort;
import io.grpc.Status;

import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.client.StockpileClientOptions;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class NodeClient implements AutoCloseable {

    @WillCloseWhenClosed
    private final GrpcTransport transport;
    private final MetricRegistry registry;

    NodeClient(HostAndPort address, StockpileClientOptions options) {
        this.transport = new GrpcTransport(address, options.getGrpcOptions());
        this.registry = options.getGrpcOptions().getMetricRegistry();
    }

    HostAndPort getAddress() {
        return transport.getAddress();
    }

    <ReqT, RespT> CompletableFuture<RespT> unaryCall(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request) {
        long deadline = endpoint.getDeadline(request);
        return transport.unaryCall(endpoint.getMethod(), request, deadline)
                .handle((response, throwable) -> {
                    EStockpileStatusCode statusCode;
                    if (throwable != null) {
                        RequestStatus status = classifyError(throwable);
                        statusCode = status.getCode();
                        response = endpoint.handleError(status.getCode(), status.getDetails());
                    } else {
                        statusCode = endpoint.getStatusCode(response);
                    }

            registry.rate("stockpile.client.call.status",
                    Labels.of("endpoint", endpoint.getMethod().getFullMethodName(),
                            "code", statusCode.name()))
                    .inc();
            return response;
        });
    }

    private RequestStatus classifyError(Throwable e) {
        final Throwable unwrapped = CompletableFutures.unwrapCompletionException(e);
        if (unwrapped instanceof StockpileRuntimeException) {
            EStockpileStatusCode code = ((StockpileRuntimeException) unwrapped).getCode();
            return new RequestStatus(code, unwrapped.getMessage());
        }

        Status status = Status.fromThrowable(unwrapped);
        switch (status.getCode()) {
            case RESOURCE_EXHAUSTED:
                return new RequestStatus(EStockpileStatusCode.RESOURCE_EXHAUSTED, status.toString());
            case DEADLINE_EXCEEDED:
                return new RequestStatus(EStockpileStatusCode.DEADLINE_EXCEEDED, status.toString());
            case UNAVAILABLE:
            case CANCELLED:
                return new RequestStatus(EStockpileStatusCode.NODE_UNAVAILABLE, status.toString());
            case UNKNOWN:
                if (status.getCause() instanceof ClosedChannelException) {
                    return new RequestStatus(EStockpileStatusCode.NODE_UNAVAILABLE, status.toString());
                }
                // fall through
            default:
                return new RequestStatus(EStockpileStatusCode.INTERNAL_ERROR, status.toString());
        }
    }

    @Override
    public String toString() {
        return "NodeClient{" +
                "address=" + getAddress() +
                '}';
    }

    @Override
    public void close() {
        transport.close();
    }
}
