package ru.yandex.stockpile.client.impl;


import com.google.common.collect.Range;

import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.TServerStatusResponse;
import ru.yandex.stockpile.api.TShardStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardsState {
    private final String node;
    private final Shard[] shards;
    private final int totalShards;
    private final AvailabilityStatus availability;
    private final long createdAt;
    private final Range<Integer> compressFormat;

    private ShardsState(String node, Shard[] shards, int totalShards, AvailabilityStatus availability, Range<Integer> format, long createdAt) {
        this.node = node;
        this.shards = shards;
        this.totalShards = totalShards;
        this.availability = availability;
        this.createdAt = createdAt;
        this.compressFormat = format;
    }

    public String getFqdn() {
        return node;
    }

    public Shard[] getShards() {
        return shards;
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public Range<Integer> getCompressFormat() {
        return compressFormat;
    }

    public int getTotalShards() {
        return totalShards;
    }

    public static ShardsState init(String node) {
        return new ShardsState(
            node,
            new Shard[0],
            0,
            new AvailabilityStatus(0, node + " not initialized yet"),
            Range.openClosed(0,0),
            System.currentTimeMillis());
    }

    public static ShardsState of(String node, long createdAt, TServerStatusResponse response) {
        final Shard[] shards;
        int ready = 0;
        int index = 0;
        shards = new Shard[response.getShardStatusCount()];
        for (TShardStatus status : response.getShardStatusList()) {
            Shard shard = new Shard(node, status.getShardId(), status.getReady(), status.getReadyWrite(), status.getReadyRead(), createdAt);
            shards[index++] = shard;
            if (shard.isReady()) {
                ready++;
            }
        }

        int totalShards = response.getTotalShardCount();
        AvailabilityStatus availabilityStatus = calculateAvailability(node, shards.length, ready);
        Range<Integer> format = Range.closed(response.getOlderSupportBinaryVersion(), response.getLatestSupportBinaryVersion());
        return new ShardsState(node, shards, totalShards, availabilityStatus, format, createdAt);
    }

    private static AvailabilityStatus calculateAvailability(String node, int total, int readyCount) {
        if (readyCount == total) {
            return AvailabilityStatus.AVAILABLE;
        }

        String details = node + ": ready shards " + readyCount + "/" + total;
        double availability = (double) readyCount / (double) total;
        return new AvailabilityStatus(availability, details);
    }

    public AvailabilityStatus getAvailability() {
        return availability;
    }
}
