package ru.yandex.stockpile.client.impl;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.WillCloseWhenClosed;

import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TServerStatusRequest;
import ru.yandex.stockpile.client.StockpileClientOptions;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileNode implements AutoCloseable {
    @WillCloseWhenClosed
    private final NodeClient client;
    private final PingActorRunner actor;
    private volatile ShardsState state;

    public StockpileNode(HostAndPort address, StockpileClientOptions opts, ScheduledExecutorService timer) {
        this.client = new NodeClient(address, opts);
        this.state = ShardsState.init(getFqdn());
        this.actor = PingActorRunner.newBuilder()
                .operation("refresh_stockpile_shards_state_at_" + getFqdn())
                .pingInterval(Duration.ofMillis(opts.getMetadataExpireMs() / 2))
                .backoffMaxDelay(Duration.ofMillis(opts.getMetadataExpireMs()))
                .timer(timer)
                .executor(opts.getGrpcOptions().getRpcExecutor().orElse(MoreExecutors.directExecutor()))
                .onPing(this::act)
                .build();
        actor.forcePing();
    }

    public ShardsState getState() {
        return state;
    }

    public NodeClient getClient() {
        return client;
    }

    private CompletableFuture<Void> act(int attempt) {
        var now = System.currentTimeMillis();
        return client.unaryCall(EndpointDescriptors.METHOD_SERVER_STATUS, TServerStatusRequest.getDefaultInstance())
                .thenAccept(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        throw new StockpileRuntimeException(response.getStatus(), response.getStatusMessage());
                    }

                    state = ShardsState.of(getFqdn(), now, response);
                });
    }

    public CompletableFuture<Void> forceUpdate() {
        return actor.forcePing();
    }

    public String getFqdn() {
        return client.getAddress().getHost();
    }

    @Override
    public void close() {
        actor.close();
        client.close();
    }
}
