package ru.yandex.stockpile.client.mem;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import com.google.common.base.Throwables;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufOutputStream;

import ru.yandex.stockpile.api.TShardCommandRequest;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class AccumulatedShardCommand implements AutoCloseable {
    private final int shardId;
    /**
     * UNIX time millis after that request can be rejected with timeout, zero means that timeout not applied at all
     */
    private final long deadlineMillis;

    @WillCloseWhenClosed
    private final ByteBuf readOnlyBuffer;
    private final int countCommands;

    public AccumulatedShardCommand(TShardCommandRequest proto) {
        this.shardId = proto.getShardId();
        this.deadlineMillis = proto.getDeadline();
        this.countCommands = proto.getCommandsCount();
        ByteBuf buffer = ByteBufAllocator.DEFAULT.buffer(proto.getSerializedSize(), proto.getSerializedSize());
        try {
            proto.writeTo(new ByteBufOutputStream(buffer));
        } catch (Throwable e) {
            buffer.release();
            throw Throwables.propagate(e);
        }

        this.readOnlyBuffer = buffer.asReadOnly();
    }

    AccumulatedShardCommand(int shardId, long deadlineMillis, @WillCloseWhenClosed ByteBuf readOnlyBuffer, int countCommands) {
        this.shardId = shardId;
        this.deadlineMillis = deadlineMillis;
        this.readOnlyBuffer = readOnlyBuffer.asReadOnly();
        this.countCommands = countCommands;
    }

    public int getShardId() {
        return shardId;
    }

    public long getDeadlineMillis() {
        return deadlineMillis;
    }

    public int getCountCommands() {
        return countCommands;
    }

    public ByteBuf getReadOnlyBuffer() {
        return readOnlyBuffer;
    }

    @Override
    public void close() {
        readOnlyBuffer.release();
    }
}
