package ru.yandex.stockpile.client.mem;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TWriteRequest;

/**
 * @author Vladimir Gordiychuk
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class CrossShardCommandAccumulator implements AutoCloseable {
    private final Map<Integer, ShardCommandAccumulator> shardIdToAccummulator = new HashMap<>();

    public void append(TWriteRequest request) {
        getOrCreateAccumulator(request.getMetricId()).append(request);
    }

    public void append(TCompressedWriteRequest request) {
        getOrCreateAccumulator(request.getMetricId()).append(request);
    }

    public void append(DeleteMetricRequest request) {
        getOrCreateAccumulator(request.getMetricId()).append(request);
    }

    public void append(DeleteMetricDataRequest request) {
        getOrCreateAccumulator(request.getMetricId()).append(request);
    }

    private ShardCommandAccumulator getOrCreateAccumulator(MetricId metricId) {
        int shardId = metricId.getShardId();
        if (shardId == 0) {
            throw new IllegalArgumentException("shardId can't be zero for MetricId: " + metricId);
        }

        ShardCommandAccumulator result = shardIdToAccummulator.get(shardId);
        if (result == null) {
            result = new ShardCommandAccumulator(shardId);
            shardIdToAccummulator.put(shardId, result);
        }

        return result;
    }

    /**
     * Remember write position to be able reset to it.
     */
    public void markWriteIndex() {
        for (ShardCommandAccumulator accumulator : shardIdToAccummulator.values()) {
            accumulator.markWriteIndex();
        }
    }

    /**
     * Reset write index to marked position
     */
    public void resetWriterIndex() {
        Iterator<ShardCommandAccumulator> it = shardIdToAccummulator.values().iterator();
        while (it.hasNext()) {
            ShardCommandAccumulator accumulator = it.next();
            accumulator.resetWriterIndex();
            if (accumulator.getCountCommands() == 0) {
                it.remove();
            }
        }
    }

    public long countBytes() {
        long sum = 0;
        for (ShardCommandAccumulator accumulator : shardIdToAccummulator.values()) {
            sum += accumulator.getCountBytes();
        }

        return sum;
    }

    public Map<Integer, AccumulatedShardCommand> build() {
        return buildWithDeadline(0);
    }

    public Map<Integer, AccumulatedShardCommand> buildWithDeadline(long deadlineMillis) {
        Map<Integer, AccumulatedShardCommand> data = new HashMap<>();
        for (Map.Entry<Integer, ShardCommandAccumulator> entry : shardIdToAccummulator.entrySet()) {
            ShardCommandAccumulator accumulator = entry.getValue();
            data.put(entry.getKey(), accumulator.buildWithDeadline(deadlineMillis));
        }

        shardIdToAccummulator.clear();
        return data;
    }

    @Override
    public void close() {
        for (ShardCommandAccumulator accumulator : shardIdToAccummulator.values()) {
            accumulator.close();
        }

        shardIdToAccummulator.clear();
    }
}
