package ru.yandex.stockpile.client.mem;

import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.google.common.base.Throwables;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.Message;
import com.google.protobuf.WireFormat;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufOutputStream;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TWriteRequest;

/**
 * Build binary form of {@link ru.yandex.stockpile.api.TShardCommandRequest}
 *
 * @author Vladimir Gordiychuk
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class ShardCommandAccumulator implements AutoCloseable {
    private final int shardId;
    private final ByteBuf buffer;
    private final CodedOutputStream output;
    private int countBytes;
    private int countCommands;
    private boolean closed;

    private int markedCountBytes;
    private int markedCountCommands;

    public ShardCommandAccumulator(int shardId) {
        this.shardId = shardId;
        this.buffer = ByteBufAllocator.DEFAULT.compositeBuffer();
        try {
            this.countBytes = CodedOutputStream.computeUInt32Size(TShardCommandRequest.SHARDID_FIELD_NUMBER, shardId);
            this.output = CodedOutputStream.newInstance(new ByteBufOutputStream(buffer));
            this.output.writeUInt32(TShardCommandRequest.SHARDID_FIELD_NUMBER, shardId);
            this.buffer.markWriterIndex();
        } catch (Throwable e) {
            buffer.release();
            throw Throwables.propagate(e);
        }
    }

    /**
     * Remember write position to be able reset to it.
     */
    public void markWriteIndex() {
        try {
            this.output.flush();
            this.buffer.markWriterIndex();
            this.markedCountBytes = this.countBytes;
            this.markedCountCommands = this.countCommands;
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    /**
     * Reset write index to marked position
     */
    public void resetWriterIndex() {
        try {
            this.output.flush();
            this.buffer.resetWriterIndex();
            this.countBytes = this.markedCountBytes;
            this.countCommands = this.markedCountCommands;
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public void append(TWriteRequest request) {
        ensureNotClosed();
        ensureShardValid(request.getMetricId(), request);
        appendCommand(TCommandRequest.RequestCase.WRITE, request);
    }

    public void append(TCompressedWriteRequest request) {
        ensureNotClosed();
        ensureShardValid(request.getMetricId(), request);
        appendCommand(TCommandRequest.RequestCase.COMPRESSEDWRITE, request);
    }

    public void append(DeleteMetricDataRequest request) {
        ensureNotClosed();
        ensureShardValid(request.getMetricId(), request);
        appendCommand(TCommandRequest.RequestCase.DELETE_METRIC_DATA, request);
    }

    public void append(DeleteMetricRequest request) {
        ensureNotClosed();
        ensureShardValid(request.getMetricId(), request);
        appendCommand(TCommandRequest.RequestCase.DELETE_METRIC, request);
    }

    private void appendCommand(TCommandRequest.RequestCase type, Message value) {
        try {
            int commandSize = CodedOutputStream.computeMessageSize(type.getNumber(), value);
            countCommands++;
            countBytes += CodedOutputStream.computeTagSize(TShardCommandRequest.COMMANDS_FIELD_NUMBER);
            countBytes += CodedOutputStream.computeUInt32SizeNoTag(commandSize);
            countBytes += commandSize;

            output.writeTag(TShardCommandRequest.COMMANDS_FIELD_NUMBER, WireFormat.WIRETYPE_LENGTH_DELIMITED);
            output.writeUInt32NoTag(commandSize);
            output.writeMessage(type.getNumber(), value);
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private void ensureNotClosed() {
        if (closed) {
            throw new IllegalStateException("Already closed accumulator for shard " + shardId);
        }
    }

    private void ensureShardValid(MetricId metricId, Message request) {
        int requestShardId = metricId.getShardId();
        if (requestShardId != 0 && requestShardId != shardId) {
            throw new IllegalArgumentException("ShardId " + shardId + " not connected with request " + request);
        }
    }

    public AccumulatedShardCommand build() {
        return buildWithDeadline(0);
    }

    public AccumulatedShardCommand buildWithDeadline(long deadlineMillis) {
        try {
            if (deadlineMillis != 0) {
                output.writeUInt64(TShardCommandRequest.DEADLINE_FIELD_NUMBER, deadlineMillis);
            }
            output.flush();
            AccumulatedShardCommand result = new AccumulatedShardCommand(shardId, deadlineMillis, buffer.asReadOnly(), countCommands);
            closed = true;
            return result;
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public int getCountBytes() {
        return countBytes;
    }

    public int getCountCommands() {
        return countCommands;
    }

    public boolean isClosed() {
        return closed;
    }

    @Override
    public void close() {
        if (!closed) {
            closed = true;
            buffer.release();
        }
    }
}
