package ru.yandex.stockpile.client.shard;

import java.util.Random;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.AbstractLongComparator;
import it.unimi.dsi.fastutil.longs.LongArrays;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.concurrent.ThreadUtils;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileLocalId implements MemMeasurable {

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileLocalId.class);

    public static final long MIN = 0;
    public static final long MAX = -1;

    public static final long MIN_VALID = 0x1_0000_0000L;

    private final long id;

    public StockpileLocalId(long id) {
        validate(id);
        this.id = id;
    }

    public long getId() {
        return id;
    }

    public static boolean isValid(long localId) {
        return localId != 0 && localId != -1;
    }

    public static boolean isValidNew(long localId) {
        return isValid(localId) && Long.compareUnsigned(localId, MIN_VALID) >= 0;
    }

    public static void validate(long localId) {
        if (!isValid(localId)) {
            throw new IllegalArgumentException("invalid local id: " + localId);
        }
    }

    public static int compare(long a, long b) {
        return Long.compareUnsigned(a, b);
    }

    public static long min(long[] array) {
        long min = MAX;
        for (long l : array) {
            if (Long.compareUnsigned(l, min) < 0) {
                min = l;
            }
        }
        return min;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        StockpileLocalId that = (StockpileLocalId) o;

        return id == that.id;

    }

    @Override
    public int hashCode() {
        return Long.hashCode(id);
    }

    @Override
    public String toString() {
        return toString(id);
    }

    public static String toString(long localId) {
        return Long.toUnsignedString(localId);
    }

    public static long random(Random random) {
        for (;;) {
            long r = random.nextLong();
            if (isValidNew(r)) {
                return r;
            }
        }
    }

    public static long random() {
        return random(ThreadUtils.currentThreadLocalRandom());
    }

    public static void sort(long[] localIds) {
        LongArrays.quickSort(localIds, new AbstractLongComparator() {
            @Override
            public int compare(long a, long b) {
                return StockpileLocalId.compare(a, b);
            }
        });
    }

    public static long parse(String s) {
        if (s.startsWith("-")) {
            return Long.parseLong(s);
        } else {
            return Long.parseUnsignedLong(s, 10);
        }
    }

    public static StockpileLocalId parseObject(String s) {
        return new StockpileLocalId(parse(s));
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE;
    }
}
