package ru.yandex.stockpile.client.shard;

import java.util.Random;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileMetricId implements Comparable<StockpileMetricId> {
    private final int shardId;
    private final long localId;

    public StockpileMetricId(int shardId, long localId) {
        StockpileShardId.validate(shardId);
        StockpileLocalId.validate(localId);
        this.shardId = shardId;
        this.localId = localId;
    }

    public int getShardId() {
        return shardId;
    }

    public long getLocalId() {
        return localId;
    }

    @Override
    public String toString() {
        return toString(shardId, localId);
    }

    @Nonnull
    public static String toString(int shardId, long localId) {
        return StockpileShardId.toString(shardId) + "/" + StockpileLocalId.toString(localId);
    }

    public static StockpileMetricId parse(String value) {
        int index = value.indexOf('/');
        if (index == -1) {
            throw new IllegalArgumentException("value does not contains delimiter '/': " + value);
        }
        int shardId = StockpileShardId.parse(value.substring(0, index));
        long localId = StockpileLocalId.parse(value.substring(index + 1));
        return new StockpileMetricId(shardId, localId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        StockpileMetricId that = (StockpileMetricId) o;

        if (shardId != that.shardId) return false;
        return localId == that.localId;

    }

    @Override
    public int hashCode() {
        int result = shardId;
        result = 31 * result + (int) (localId ^ (localId >>> 32));
        return result;
    }

    @Override
    public int compareTo(@Nonnull StockpileMetricId o) {
        if (shardId != o.shardId) {
            return Integer.compare(shardId, o.shardId);
        }
        if (localId != o.localId) {
            return StockpileLocalId.compare(localId, o.localId);
        }
        return 0;
    }

    public static StockpileMetricId random(Random random) {
        return new StockpileMetricId(StockpileShardId.random(random), StockpileLocalId.random(random));
    }

    public static StockpileMetricId random() {
        return new StockpileMetricId(StockpileShardId.random(), StockpileLocalId.random());
    }
}
