package ru.yandex.stockpile.client.shard;

import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileShardId {
    private static final int STOCKPILE_SHARD_COUNT = 4 << 10;
    public static final int FIRST_SHARD_ID = 1;
    private static final int MAX_VALID_SHARD_ID = 1_000_000;
    public static final int INVALID_SHARD_ID = FIRST_SHARD_ID - 1;

    private final int id;

    public StockpileShardId(int id) {
        validate(id);
        this.id = id;
    }

    public int getId() {
        return id;
    }

    public static void validate(int shardId) {
        if (shardId < FIRST_SHARD_ID || shardId > MAX_VALID_SHARD_ID) {
            throw new IllegalArgumentException("wrong shard id: " + shardId);
        }
    }

    @VisibleForTesting
    @Deprecated
    public static int random() {
        return random(ThreadUtils.currentThreadLocalRandom());
    }

    @VisibleForTesting
    @Deprecated
    public static int random(Random random) {
        return random(STOCKPILE_SHARD_COUNT, random);
    }

    public static int random(int shardCount) {
        return random(shardCount, ThreadLocalRandom.current());
    }

    public static int random(int shardCount, Random random) {
        if (shardCount <= 0) {
            throw new IllegalArgumentException("Invalid shard count: " + shardCount);
        }
        return FIRST_SHARD_ID + random.nextInt(shardCount);
    }

    public static int parse(String s) {
        int r = Integer.parseInt(s, 10);
        validate(r);
        return r;
    }

    @Nonnull
    public static StockpileShardId parseObject(String x) {
        return new StockpileShardId(parse(x));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        StockpileShardId that = (StockpileShardId) o;

        return id == that.id;

    }

    @Override
    public int hashCode() {
        return id;
    }

    @Nonnull
    public static String toString(int id) {
        return Integer.toString(id);
    }

    @Override
    public String toString() {
        return toString(id);
    }

    public static int compare(int a, int b) {
        return Integer.compare(a, b);
    }
}
