package ru.yandex.stockpile.client.util;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.TPoint;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class ChunkEncoder {
    private ChunkEncoder() {
    }

    public static List<TimeSeries.Chunk> encode(Encoder encoder, List<TPoint> points) {
        if (points.isEmpty()) {
            return Collections.emptyList();
        }

        Map<Instant, List<TPoint>> groupedByChunk = points.stream()
                .sorted(Comparator.comparingLong(TPoint::getTimestampsMillis))
                .collect(Collectors.groupingBy(point -> Instant.ofEpochMilli(point.getTimestampsMillis()).truncatedTo(ChronoUnit.DAYS)));

        List<TimeSeries.Chunk> chunks = new ArrayList<>(groupedByChunk.size());
        for (Map.Entry<Instant, List<TPoint>> entry : groupedByChunk.entrySet()) {
            byte[] compressed = encoder.encode(entry.getValue());
            Instant from = entry.getKey();
            Instant to = from.plus(1, ChronoUnit.DAYS).minus(1, ChronoUnit.MILLIS);

            TimeSeries.Chunk chunk = TimeSeries.Chunk.newBuilder()
                    .setPointCount(entry.getValue().size())
                    .setFromMillis(from.toEpochMilli())
                    .setToMillis(to.toEpochMilli())
                    .setContent(ByteString.copyFrom(compressed))
                    .build();

            chunks.add(chunk);
        }

        return chunks;
    }

    public static TimeSeries encodeTs(Encoder encoder, List<TPoint> points) {
        if (points.isEmpty()) {
            return TimeSeries.getDefaultInstance();
        }

        Map<Instant, List<TPoint>> groupedByChunk = points.stream()
                .sorted(Comparator.comparingLong(TPoint::getTimestampsMillis))
                .collect(Collectors.groupingBy(point -> Instant.ofEpochMilli(point.getTimestampsMillis()).truncatedTo(ChronoUnit.DAYS)));

        List<TimeSeries.Chunk> chunks = new ArrayList<>(groupedByChunk.size());
        for (Map.Entry<Instant, List<TPoint>> entry : groupedByChunk.entrySet()) {
            byte[] compressed = encoder.encode(entry.getValue());
            Instant from = entry.getKey();
            Instant to = from.plus(1, ChronoUnit.DAYS).minus(1, ChronoUnit.MILLIS);

            TimeSeries.Chunk chunk = TimeSeries.Chunk.newBuilder()
                    .setPointCount(entry.getValue().size())
                    .setFromMillis(from.toEpochMilli())
                    .setToMillis(to.toEpochMilli())
                    .setContent(ByteString.copyFrom(compressed))
                    .build();

            chunks.add(chunk);
        }

        return TimeSeries.newBuilder()
                .addAllChunks(chunks)
                .build();
    }

    public static List<TPoint> decode(Encoder encoder, List<TimeSeries.Chunk> chunks) {
        return chunks.stream().flatMap(chunk -> {
            byte[] content = chunk.getContent().toByteArray();
            List<TPoint> points = encoder.decode(content);
            return points.stream();
        }).collect(Collectors.toList());
    }

}
