package ru.yandex.stockpile.client.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.ExtensionRegistryLite;

import ru.yandex.stockpile.api.TPoint;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public enum Encoder {
    V41_OLDER(41) {
        @Override
        public byte[] encode(List<TPoint> points) {
            try {
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                for (TPoint point : points) {
                    point.writeDelimitedTo(out);
                }

                return out.toByteArray();
            } catch (IOException e) {
                throw new EncoderRuntimeException(e);
            }
        }

        @Override
        public List<TPoint> decode(byte[] bytes) {
            try {
                ByteArrayInputStream input = new ByteArrayInputStream(bytes);
                List<TPoint> result = new ArrayList<>();
                while (input.available() > 0) {
                    TPoint point = TPoint.parseDelimitedFrom(input);
                    result.add(point);
                }

                return result;
            } catch (IOException e) {
                throw new EncoderRuntimeException(e);
            }
        }
    },

    V42_POINTS_CAPACITY(42) {
        @Override
        public byte[] encode(List<TPoint> points) {
            try {
                ByteArrayOutputStream result = new ByteArrayOutputStream();
                CodedOutputStream out = CodedOutputStream.newInstance(result);
                out.writeUInt32NoTag(points.size());
                for (TPoint point : points) {
                    out.writeMessageNoTag(point);
                }
                out.flush();
                return result.toByteArray();
            } catch (IOException e) {
                throw new EncoderRuntimeException(e);
            }

        }

        @Override
        public List<TPoint> decode(byte[] bytes) {
            try {
                CodedInputStream input = CodedInputStream.newInstance(bytes);
                int size = input.readUInt32();
                List<TPoint> result = new ArrayList<>(size);
                while (!input.isAtEnd()) {
                    TPoint point = input.readMessage(TPoint.parser(), ExtensionRegistryLite.getEmptyRegistry());
                    result.add(point);
                }

                return result;
            } catch (IOException e) {
                throw new EncoderRuntimeException(e);
            }
        }
    }
    ;

    private final int version;

    Encoder(int version) {
        this.version = version;
    }

    public abstract byte[] encode(List<TPoint> points);

    public abstract List<TPoint> decode(byte[] bytes);

    public int getVersion() {
        return version;
    }

    public static Optional<Encoder> valueOf(int version) {
        for (Encoder encoder : Encoder.values()) {
            if (encoder.getVersion() == version) {
                return Optional.of(encoder);
            }
        }

        return Optional.empty();
    }
}
