package ru.yandex.stockpile.client.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EColumnFlag;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TTimeseriesUncompressed;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.ColumnFlagMask;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
class InMemoryShard {
    private final int shardId;
    private final boolean writeToDevNull;
    private final AtomicLong recordCount = new AtomicLong(0);
    private final AtomicLong metricCount = new AtomicLong(0);
    private ConcurrentMap<Long, CreateMetricRequest> localIdToMetaData = new ConcurrentHashMap<>();
    private ConcurrentMap<Long, List<TPoint>> localIdToPoints = new ConcurrentHashMap<>();

    InMemoryShard(int shardId, boolean writeToDevNull) throws IOException {
        this.shardId = shardId;
        this.writeToDevNull = writeToDevNull;
    }

    CreateMetricResponse createMetric(CreateMetricRequest request) {
        MetricId metricId = makeMetricId(request);

        CreateMetricResponse.Builder responseBuilder = CreateMetricResponse.newBuilder();
        CreateMetricRequest previous = localIdToMetaData.putIfAbsent(metricId.getLocalId(), request);
        if (previous != null) {
            return CreateMetricResponse.newBuilder().setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.METRIC_ALREADY_EXISTS)
                    .build();
        }

        metricCount.incrementAndGet();
        return responseBuilder
                .setStatus(EStockpileStatusCode.OK)
                .setMetricId(metricId)
                .build();
    }

    TWriteResponse write(TWriteRequest request) {
        TWriteResponse.Builder responseBuilder = TWriteResponse.newBuilder().setMetricId(request.getMetricId());

        if (!isTimeSeriesExists(request.getMetricId())) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        try {
            writeExistTimeSeries(request.getMetricId().getLocalId(), request.getPointsList());
            return responseBuilder.setStatus(EStockpileStatusCode.OK).build();
        } catch (IOException e) {
            return responseBuilder.setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                    .setStatusMessage(Throwables.getStackTraceAsString(e))
                    .build();
        }
    }

    TCompressedWriteResponse writeCompressed(TCompressedWriteRequest request) {
        TCompressedWriteResponse.Builder responseBuilder = TCompressedWriteResponse.newBuilder().setMetricId(request.getMetricId());

        if (!isTimeSeriesExists(request.getMetricId())) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        Optional<Encoder> encoder = Encoder.valueOf(request.getBinaryVersion());
        if (!encoder.isPresent()) {
            return responseBuilder
                    .setStatus(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT)
                    .setStatusMessage("Supported binary version [41, 42] but specified " + request.getBinaryVersion())
                    .build();
        }

        try {
            List<TPoint> points = ChunkEncoder.decode(encoder.get(), request.getChunksList());
            writeExistTimeSeries(request.getMetricId().getLocalId(), points);
            return responseBuilder.setStatus(EStockpileStatusCode.OK).build();
        } catch (IOException e) {
            return responseBuilder.setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                    .setStatusMessage(Throwables.getStackTraceAsString(e))
                    .build();
        } catch (EncoderRuntimeException e) {
            return responseBuilder.setStatus(EStockpileStatusCode.CORRUPTED_BINARY_DATA)
                    .setStatusMessage(Throwables.getStackTraceAsString(e))
                    .build();
        }
    }

    TReadResponse read(TReadRequest request) {
        TReadResponse.Builder responseBuilder = TReadResponse.newBuilder();
        responseBuilder.setMetricId(request.getMetricId());
        responseBuilder.setCookie(request.getCookie());

        long localId = request.getMetricId().getLocalId();
        CreateMetricRequest metaData = localIdToMetaData.get(localId);
        if (metaData == null) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        responseBuilder.setColumnMask(columnSetMask(metaData.getType()));
        List<TPoint> points = localIdToPoints.getOrDefault(localId, Collections.emptyList());
        responseBuilder.addAllPoints(transformPoints(request, points));
        responseBuilder.setStatus(EStockpileStatusCode.OK);

        return responseBuilder.build();
    }

    TUncompressedReadManyResponse readUncompressedMany(TReadManyRequest request) {
        return request.getLocalIdsList()
                .stream()
                .map(localId -> {
                    CreateMetricRequest metaData = localIdToMetaData.get(localId);
                    if (metaData == null) {
                        throw new IllegalStateException(EStockpileStatusCode.METRIC_NOT_FOUND.name());
                    }

                    List<TPoint> points = localIdToPoints.getOrDefault(localId, Collections.emptyList());
                    return MetricData.newBuilder()
                            .setShardId(request.getShardId())
                            .setLocalId(localId)
                            .setUncompressed(TTimeseriesUncompressed.newBuilder()
                                    .setColumnMask(columnSetMask(metaData.getType()))
                                    .addAllPoints(transformPoints(request.getFromMillis(), request.getToMillis(), points))
                                    .build())
                            .build();
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(),
                        list -> TUncompressedReadManyResponse.newBuilder()
                                .setStatus(EStockpileStatusCode.OK)
                                .addAllMetrics(list)
                                .build()));
    }

    TCompressedReadManyResponse readCompressedMany(TReadManyRequest request) {
        Optional<Encoder> encoder = Encoder.valueOf(request.getBinaryVersion());
        if (!encoder.isPresent()) {
            return TCompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT)
                    .setStatusMessage("Supported binary version [41, 42] but specified " + request.getBinaryVersion())
                    .build();
        }

        return request.getLocalIdsList()
                .stream()
                .map(localId -> {
                    CreateMetricRequest metaData = localIdToMetaData.get(localId);
                    if (metaData == null) {
                        throw new IllegalStateException(EStockpileStatusCode.METRIC_NOT_FOUND.name());
                    }

                    List<TPoint> points = localIdToPoints.getOrDefault(localId, Collections.emptyList());
                    List<TPoint> filtered = transformPoints(request.getFromMillis(), request.getToMillis(), points);
                    return MetricData.newBuilder()
                            .setShardId(request.getShardId())
                            .setLocalId(localId)
                            .setCompressed(ChunkEncoder.encodeTs(encoder.get(), filtered))
                            .build();
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(),
                        list -> TCompressedReadManyResponse.newBuilder()
                                .setStatus(EStockpileStatusCode.OK)
                                .addAllMetrics(list)
                                .build()));
    }

    TCompressedReadResponse readCompressed(TReadRequest request) {
        TCompressedReadResponse.Builder responseBuilder = TCompressedReadResponse.newBuilder();
        responseBuilder.setMetricId(request.getMetricId());
        responseBuilder.setCookie(request.getCookie());

        long localId = request.getMetricId().getLocalId();
        CreateMetricRequest metaData = localIdToMetaData.get(localId);
        if (metaData == null) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        Optional<Encoder> encoder = Encoder.valueOf(request.getBinaryVersion());
        if (!encoder.isPresent()) {
            return responseBuilder
                    .setStatus(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT)
                    .setStatusMessage("Supported binary version [41, 42] but specified " + request.getBinaryVersion())
                    .build();
        }

        responseBuilder.setType(metaData.getType());
        responseBuilder.setBinaryVersion(request.getBinaryVersion());
        List<TPoint> points = localIdToPoints.getOrDefault(localId, Collections.emptyList());
        responseBuilder.addAllChunks(ChunkEncoder.encode(encoder.get(), transformPoints(request, points)));
        responseBuilder.setStatus(EStockpileStatusCode.OK);

        return responseBuilder.build();
    }

    DeleteMetricResponse delete(DeleteMetricRequest request) {
        DeleteMetricResponse.Builder responseBuilder = DeleteMetricResponse.newBuilder();
        responseBuilder.setMetricId(request.getMetricId());

        long localId = request.getMetricId().getLocalId();
        CreateMetricRequest metaData = localIdToMetaData.remove(localId);
        if (metaData == null) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        localIdToPoints.remove(localId);

        return responseBuilder
                .setStatus(EStockpileStatusCode.OK)
                .build();
    }

    DeleteMetricDataResponse deleteMetricData(DeleteMetricDataRequest request) {
        DeleteMetricDataResponse.Builder responseBuilder = DeleteMetricDataResponse.newBuilder();
        responseBuilder.setMetricId(request.getMetricId());

        if (!isTimeSeriesExists(request.getMetricId())) {
            return responseBuilder.setStatus(EStockpileStatusCode.METRIC_NOT_FOUND).build();
        }

        long localId = request.getMetricId().getLocalId();

        List<TPoint> points = localIdToPoints.getOrDefault(localId, Collections.emptyList());
        List<TPoint> filteredPoints = new ArrayList<>();
        for (TPoint point : points) {
            boolean leftBound = request.getFromMillis() == 0 || point.getTimestampsMillis() >= request.getFromMillis();
            boolean rightBound = request.getToMillis() == 0 || point.getTimestampsMillis() < request.getToMillis();
            if (leftBound && rightBound) {
                continue;
            }

            filteredPoints.add(point);
        }

        localIdToPoints.put(localId, filteredPoints);
        return responseBuilder
                .setStatus(EStockpileStatusCode.OK)
                .build();
    }

    private List<TPoint> transformPoints(TReadRequest request, List<TPoint> points) {
        return transformPoints(request.getFromMillis(), request.getToMillis(), points);
    }

    private List<TPoint> transformPoints(long fromMillis, long toMillis, List<TPoint> points) {
        if (fromMillis == 0 && toMillis == 0) {
            return points;
        }

        return points.stream()
                .sorted(Comparator.comparingLong(TPoint::getTimestampsMillis))
                .filter(point -> fromMillis == 0 || point.getTimestampsMillis() >= fromMillis)
                .filter(point -> toMillis == 0 || point.getTimestampsMillis() < toMillis)
                .collect(Collectors.toList());
    }

    long getRecordCount() {
        return recordCount.get();
    }

    long getMetricCount() {
        return metricCount.get();
    }

    private MetricId makeMetricId(CreateMetricRequest request) {
        MetricId.Builder metricIdBuilder = MetricId.newBuilder(request.getMetricId());

        if (metricIdBuilder.getShardId() == 0) {
            metricIdBuilder.setShardId(shardId);
        }

        if (metricIdBuilder.getLocalId() == 0) {
            metricIdBuilder.setLocalId(ThreadLocalRandom.current().nextLong());
        }

        return metricIdBuilder.build();
    }

    private void writeExistTimeSeries(long localId, Iterable<TPoint> points) throws IOException {
        localIdToPoints.compute(localId, (id, exists) -> {
            List<TPoint> result = exists != null ? new ArrayList<>(exists) : new ArrayList<>();

            for (TPoint point : points) {
                if (!writeToDevNull) {
                    result.add(point);
                }
                recordCount.incrementAndGet();
            }

            return result;
        });
    }

    private boolean isTimeSeriesExists(MetricId metric) {
        return localIdToMetaData.containsKey(metric.getLocalId());
    }

    private int columnSetMask(MetricType type) {
        switch (type) {
            case DGAUGE:
            case IGAUGE:
            case COUNTER:
            case RATE:
                return ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP, EColumnFlag.COLUMN_DOUBLE);
            case ISUMMARY:
                return ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP, EColumnFlag.COLUMN_ISUMMARY);
            case DSUMMARY:
                return ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP, EColumnFlag.COLUMN_DSUMMARY);
            case HIST:
            case HIST_RATE:
                return ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP, EColumnFlag.COLUMN_HISTOGRAM);
            case METRIC_TYPE_UNSPECIFIED:
                return ColumnFlagMask.mask(EColumnFlag.COLUMN_TIMESTAMP);
            default:
                throw new UnsupportedOperationException("Unsupported type: " + type);
        }
    }

}
