package ru.yandex.stockpile.client.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.protobuf.Message;
import io.grpc.stub.StreamObserver;

import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TServerStatusRequest;
import ru.yandex.stockpile.api.TServerStatusResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TShardStatus;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.impl.ResponseProtoFieldAccessor;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class InMemoryStockpileService extends StockpileServiceGrpc.StockpileServiceImplBase {
    private final ConcurrentMap<Integer, InMemoryShard> shardIdToShard;
    private final int totalShardsCount;

    public InMemoryStockpileService(int localShards[], int totalShardsCount, boolean writeToDevNull) throws IOException {
        this.shardIdToShard = new ConcurrentHashMap<>();
        for (int shard : localShards) {
            shardIdToShard.put(shard, new InMemoryShard(shard, writeToDevNull));
        }
        this.totalShardsCount = totalShardsCount;
    }

    public boolean hasShard(int shardId) {
        return shardIdToShard.containsKey(shardId);
    }

    @Override
    public void serverStatus(TServerStatusRequest request, StreamObserver<TServerStatusResponse> responseObserver) {
        List<TShardStatus> shardStatuses = new ArrayList<>(shardIdToShard.size());
        for (Map.Entry<Integer, InMemoryShard> entry : shardIdToShard.entrySet()) {
            TShardStatus status = TShardStatus.newBuilder()
                    .setShardId(entry.getKey())
                    .setReady(true)
                    .setReadyWrite(true)
                    .setReadyRead(true)
                    .setRecordCount(entry.getValue().getRecordCount())
                    .setMetricCount(entry.getValue().getMetricCount())
                    .build();

            shardStatuses.add(status);
        }

        TServerStatusResponse response = TServerStatusResponse
                .newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .addAllShardStatus(shardStatuses)
                .setTotalShardCount(totalShardsCount)
                .setLatestSupportBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .setOlderSupportBinaryVersion(Encoder.V41_OLDER.getVersion())
                .build();

        responseObserver.onNext(response);
        responseObserver.onCompleted();
    }

    @Override
    public void createMetric(CreateMetricRequest request, StreamObserver<CreateMetricResponse> responseObserver) {
        responseObserver.onNext(createMetric(request));
        responseObserver.onCompleted();
    }

    @Override
    public void deleteMetric(DeleteMetricRequest request, StreamObserver<DeleteMetricResponse> responseObserver) {
        responseObserver.onNext(deleteMetric(request));
        responseObserver.onCompleted();
    }

    @Override
    public void deleteMetricData(DeleteMetricDataRequest request, StreamObserver<DeleteMetricDataResponse> responseObserver) {
        responseObserver.onNext(deleteMetricData(request));
        responseObserver.onCompleted();
    }

    @Override
    public void writeOne(TWriteRequest request, StreamObserver<TWriteResponse> responseObserver) {
        responseObserver.onNext(write(request));
        responseObserver.onCompleted();
    }

    @Override
    public void writeCompressedOne(TCompressedWriteRequest request, StreamObserver<TCompressedWriteResponse> responseObserver) {
        responseObserver.onNext(writeCompressed(request));
        responseObserver.onCompleted();
    }

    @Override
    public void bulkShardCommand(TShardCommandRequest request, StreamObserver<TShardCommandResponse> responseObserver) {
        responseObserver.onNext(bulkShardCommand(request));
        responseObserver.onCompleted();
    }

    @Override
    public void readOne(TReadRequest request, StreamObserver<TReadResponse> responseObserver) {
        responseObserver.onNext(read(request));
        responseObserver.onCompleted();
    }

    @Override
    public void readCompressedOne(TReadRequest request, StreamObserver<TCompressedReadResponse> responseObserver) {
        responseObserver.onNext(readCompressed(request));
        responseObserver.onCompleted();
    }

    @Override
    public void readUncompressedMany(TReadManyRequest request, StreamObserver<TUncompressedReadManyResponse> responseObserver) {
        responseObserver.onNext(readManyUncompressed(request));
        responseObserver.onCompleted();
    }

    @Override
    public void readCompressedMany(TReadManyRequest request, StreamObserver<TCompressedReadManyResponse> responseObserver) {
        responseObserver.onNext(readManyCompressed(request));
        responseObserver.onCompleted();
    }

    private boolean isShardLocal(int shardId) {
        return shardIdToShard.containsKey(shardId);
    }

    private CreateMetricResponse createMetric(CreateMetricRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (shardId == 0) {
            shardId = randomShard();
        }

        if (!isShardLocal(shardId)) {
            return CreateMetricResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).createMetric(request);
    }

    private DeleteMetricResponse deleteMetric(DeleteMetricRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return DeleteMetricResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).delete(request);
    }

    private DeleteMetricDataResponse deleteMetricData(DeleteMetricDataRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return DeleteMetricDataResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).deleteMetricData(request);
    }

    private TWriteResponse write(TWriteRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return TWriteResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).write(request);
    }

    private TCompressedWriteResponse writeCompressed(TCompressedWriteRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return TCompressedWriteResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).writeCompressed(request);
    }

    private TShardCommandResponse bulkShardCommand(TShardCommandRequest request) {
        int shardId = request.getShardId();
        if (!isShardLocal(shardId)) {
            return TShardCommandResponse.newBuilder()
                    .setShardId(shardId)
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        InMemoryShard shard = shardIdToShard.get(shardId);
        for (TCommandRequest command : request.getCommandsList()) {
            final Message response;

            switch (command.getRequestCase()) {
                case WRITE:
                    response = shard.write(command.getWrite());
                    break;
                case COMPRESSEDWRITE:
                    response = shard.writeCompressed(command.getCompressedWrite());
                    break;
                case DELETE_METRIC:
                    response = shard.delete(command.getDeleteMetric());
                    break;
                case DELETE_METRIC_DATA:
                    response = shard.deleteMetricData(command.getDeleteMetricData());
                    break;
                default:
                    throw new UnsupportedOperationException("Unsupported command: " + command);
            }


            EStockpileStatusCode statusCode = ResponseProtoFieldAccessor.getStockpileStatusCode(response);
            String statusMessage = ResponseProtoFieldAccessor.getStatusMessage(response);
            if (statusCode != EStockpileStatusCode.OK) {
                return TShardCommandResponse.newBuilder()
                        .setShardId(shardId)
                        .setStatus(statusCode)
                        .setStatusMessage(Strings.nullToEmpty(statusMessage))
                        .build();
            }
        }

        return TShardCommandResponse.newBuilder()
                .setShardId(shardId)
                .setStatus(EStockpileStatusCode.OK)
                .build();
    }

    private TReadResponse read(TReadRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return TReadResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .setCookie(request.getCookie())
                    .build();
        }

        return shardIdToShard.get(shardId).read(request);
    }

    private TCompressedReadResponse readCompressed(TReadRequest request) {
        int shardId = request.getMetricId().getShardId();
        if (!isShardLocal(shardId)) {
            return TCompressedReadResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .setCookie(request.getCookie())
                    .build();
        }

        return shardIdToShard.get(shardId).readCompressed(request);
    }

    private TUncompressedReadManyResponse readManyUncompressed(TReadManyRequest request) {
        int shardId = request.getShardId();
        if (!isShardLocal(shardId)) {
            return TUncompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).readUncompressedMany(request);
    }

    private TCompressedReadManyResponse readManyCompressed(TReadManyRequest request) {
        int shardId = request.getShardId();
        if (!isShardLocal(shardId)) {
            return TCompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.SHARD_ABSENT_ON_HOST)
                    .build();
        }

        return shardIdToShard.get(shardId).readCompressedMany(request);
    }

    private int randomShard() {
        int countShards = shardIdToShard.size();
        int index = ThreadLocalRandom.current().nextInt(countShards);
        return shardIdToShard.keySet()
                .stream()
                .skip(index)
                .findFirst()
                .orElse(0);
    }
}
