package ru.yandex.stockpile.client.writeRequest;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;
import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.writeRequest.serializers.SerializerVersion;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileShardWriteRequest implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileShardWriteRequest.class);

    private final long recordCount;
    private final long sizeBytes;
    private final EProjectId projectId;
    private final int serializerVersionId;
    private final int stockpileOwnerShardId;

    @Nullable
    private ByteBuf data;

    public StockpileShardWriteRequest(
        long recordCount,
        ByteBuf data,
        EProjectId projectId,
        int stockpileOwnerShardId,
        SerializerVersion serializerVersion)
    {
        this.recordCount = recordCount;
        this.data = data;
        this.projectId = projectId;
        this.stockpileOwnerShardId = stockpileOwnerShardId;
        this.serializerVersionId = serializerVersion.ordinal();
        this.sizeBytes = StockpileShardWriteRequestSerializer.computeSize(this);
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        var buf = data;
        if (buf != null) {
            size += buf.readableBytes();
        }
        return size;
    }

    public Long2ObjectOpenHashMap<MetricArchiveMutable> getDataByLocalId() {
        var result = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        deserializeTo(result);
        return result;
    }

    public long countRecords() {
        return recordCount;
    }

    public boolean isEmpty() {
        return recordCount == 0;
    }

    public void deserializeTo(Long2ObjectMap<MetricArchiveMutable> archiveByLocalId) {
        // We CANNOT set projectId to all archives in this collector in a cycle:
        // it may contain data from multiple sources gathered in TxWrite
        // so we have to provide it as an argument for readEventTo, and write it inside. Ugly.

        try {
            assert data != null;
            WriteRequestInputStream stream = new WriteRequestInputStream(new StockpileDeserializer(data), getVersion());
            for (long i = 0; i < recordCount; i++) {
                stream.readEventTo(archiveByLocalId, projectId, stockpileOwnerShardId);
            }
            if (!stream.atEof()) {
                throw new RuntimeException("Failed to deserialize stream");
            }
        } catch (Exception e) {
            throw new RuntimeException(
                "failed to deserialize data from: " + projectId + "/" + stockpileOwnerShardId, e);
        }
    }

    public long getSizeBytes() {
        return sizeBytes;
    }

    public EProjectId getProjectId() {
        return projectId;
    }

    public int getStockpileOwnerShardId() {
        return stockpileOwnerShardId;
    }

    public int getSerializerVersionId() {
        return serializerVersionId;
    }

    @Nullable
    public ByteBuf getData() {
        return data;
    }

    @Override
    public void close() {
        if (data != null) {
            data.release();
            data = null;
        }
    }

    public SerializerVersion getVersion() {
        return SerializerVersion.byIdOrUnknown(serializerVersionId);
    }

    public static void deserializeTo(CodedInputStream in, Long2ObjectMap<MetricArchiveMutable> archiveByLocalId) {
        StockpileShardWriteRequestSerializer.deserializeTo(in, archiveByLocalId);
    }

    public static StockpileShardWriteRequest deserialize(byte[] buf) {
        return StockpileShardWriteRequestSerializer.deserialize(buf);
    }

    public static byte[] serialize(Iterable<StockpileShardWriteRequest> requests) {
        return StockpileShardWriteRequestSerializer.serialize(requests);
    }

    public byte[] serialize() {
        return StockpileShardWriteRequestSerializer.serialize(this);
    }
}
