package ru.yandex.stockpile.client.writeRequest;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.solomon.codec.serializer.NettyStockpileSerializer;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileShardWriteRequestBuilder {

    private final EProjectId projectId;
    private final int stockpileOwnerShardId;

    private long recordCount = 0;
    private ByteBuf buffer;
    private final WriteRequestOutputStream outStream;

    public StockpileShardWriteRequestBuilder(EProjectId projectId, int stockpileOwnerShardId) {
        this.projectId = projectId;
        this.stockpileOwnerShardId = stockpileOwnerShardId;
        this.buffer = ByteBufAllocator.DEFAULT.heapBuffer();
        this.outStream = new WriteRequestOutputStream(new NettyStockpileSerializer(buffer));
    }

    public StockpileShardWriteRequest build() {
        if (buffer == null) {
            // serializer is cleaned on the first build() call and will produce unacceptable empty arrays on further calls
            throw new RuntimeException("Forbidden to build data from one builder multiple times");
        }
        var request = new StockpileShardWriteRequest(
                recordCount,
                buffer,     // takes ownership
                projectId,
                stockpileOwnerShardId,
                outStream.serializerVersion);
        buffer = null;
        return request;
    }

    public int bytesSize() {
        return buffer.readableBytes();
    }

    public void addRecord(
            long localId,
            AggrPoint point,
            int decimPolicyId,
            MetricType type)
    {
        outStream.addRecord(
            localId,
            point.columnSet,
            point,
            decimPolicyId,
            type);

        recordCount++;
    }

    public void addDeleteData(long localId) {
        outStream.addDeleteData(localId);
        recordCount++;
    }

    public void addDeleteDataWithTs(long localId, long tsMillis) {
        outStream.addDeleteDataWithTs(localId, tsMillis);
        recordCount++;
    }
}
