package ru.yandex.stockpile.client.writeRequest;

import java.io.IOException;
import java.io.UncheckedIOException;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.WireFormat;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.writeRequest.serializers.SerializerVersion;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Equivalent protobuf message:
 *
 * <pre>
 * message TWriteRequest {
 *     int64 RecordCount = 1;
 *     EProjectId ProjectId = 2;
 *     int32 Version = 3;
 *     int32 OwnerShardId = 4;
 *     bytes Data = 11;
 * }
 * </pre>
 *
 * @author Sergey Polovko
 */
final class StockpileShardWriteRequestSerializer {
    private StockpileShardWriteRequestSerializer() {}

    private static final int FIELD_RECORD_COUNT = 1;
    private static final int FIELD_PROJECT_ID = 2;
    private static final int FIELD_VERSION_ID = 3;
    private static final int FIELD_OWNER_SHARD_ID = 4;
    private static final int FIELD_DATA = 11;

    static byte[] serialize(StockpileShardWriteRequest request) {
        byte[] buf = new byte[computeSize(request)];
        var out = CodedOutputStream.newInstance(buf);
        serializeTo(request, out);
        out.checkNoSpaceLeft();
        return buf;
    }

    static byte[] serialize(Iterable<StockpileShardWriteRequest> requests) {
        int totalSize = 0;
        for (StockpileShardWriteRequest request : requests) {
            totalSize += request.getSizeBytes();
        }

        byte[] buf = new byte[totalSize];
        var out = CodedOutputStream.newInstance(buf);
        for (StockpileShardWriteRequest request : requests) {
            serializeTo(request, out);
        }
        out.checkNoSpaceLeft();
        return buf;
    }

    private static void serializeTo(StockpileShardWriteRequest request, CodedOutputStream out) {
        try {
            out.writeInt64(FIELD_RECORD_COUNT, request.countRecords());
            out.writeEnum(FIELD_PROJECT_ID, request.getProjectId().getNumber());
            out.writeInt32(FIELD_OWNER_SHARD_ID, request.getStockpileOwnerShardId());
            out.writeInt32(FIELD_VERSION_ID, request.getSerializerVersionId());

            ByteBuf data = request.getData();
            out.writeByteArray(FIELD_DATA, data.array(), data.arrayOffset() + data.readerIndex(), data.readableBytes());
        } catch (IOException e) {
            throw new UncheckedIOException("cannot serialize StockpileShardWriteRequest", e);
        }
    }

    static int computeSize(StockpileShardWriteRequest request) {
        int size = 0;
        size += CodedOutputStream.computeInt64Size(FIELD_RECORD_COUNT, request.countRecords());
        size += CodedOutputStream.computeEnumSize(FIELD_PROJECT_ID, request.getProjectId().getNumber());
        size += CodedOutputStream.computeInt32Size(FIELD_VERSION_ID, request.getSerializerVersionId());
        size += CodedOutputStream.computeInt32Size(FIELD_OWNER_SHARD_ID, request.getStockpileOwnerShardId());

        ByteBuf data = request.getData();
        size += CodedOutputStream.computeTagSize(FIELD_DATA);
        size += CodedOutputStream.computeUInt32SizeNoTag(data.readableBytes()) + data.readableBytes();
        return size;
    }

    public static StockpileShardWriteRequest deserialize(byte[] buf) {
        long recordCount = 0;
        EProjectId projectId = EProjectId.UNKNOWN;
        int stockpileOwnerShardId = 0;
        int serializerVersionId = 0;
        ByteBuf data = Unpooled.EMPTY_BUFFER;

        try {
            CodedInputStream in = CodedInputStream.newInstance(buf);
            while (!in.isAtEnd()) {
                final int tag = in.readTag();
                switch (WireFormat.getTagFieldNumber(tag)) {
                    case FIELD_RECORD_COUNT:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        recordCount = in.readInt64();
                        break;

                    case FIELD_PROJECT_ID:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        projectId = OwnerField.forNumberOrUnknown(in.readEnum());
                        break;

                    case FIELD_VERSION_ID:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        serializerVersionId = in.readInt32();
                        break;

                    case FIELD_OWNER_SHARD_ID:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        stockpileOwnerShardId = in.readInt32();
                        break;

                    case FIELD_DATA:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_LENGTH_DELIMITED);
                        data = Unpooled.wrappedBuffer(in.readByteArray());
                        break;

                    default:
                        // skip unknown field
                        in.skipField(tag);
                        break;
                }
            }
        } catch (IOException e) {
            throw new UncheckedIOException("cannot deserialize StockpileShardWriteRequest", e);
        }

        return new StockpileShardWriteRequest(
            recordCount,
            data,
            projectId,
            stockpileOwnerShardId,
            SerializerVersion.byIdOrUnknown(serializerVersionId));
    }

    public static void deserializeTo(CodedInputStream in, Long2ObjectMap<MetricArchiveMutable> archiveByLocalId) {
        try {
            EProjectId projectId = EProjectId.UNKNOWN;
            int ownerShardId = 0;
            long recordCount = 0;

            while (!in.isAtEnd()) {
                final int tag = in.readTag();
                switch (WireFormat.getTagFieldNumber(tag)) {
                    case FIELD_RECORD_COUNT:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        recordCount = in.readInt64();
                        break;

                    case FIELD_PROJECT_ID:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        projectId = OwnerField.forNumberOrUnknown(in.readEnum());
                        break;

                    case FIELD_OWNER_SHARD_ID:
                        checkArgument(WireFormat.getTagWireType(tag) == WireFormat.WIRETYPE_VARINT);
                        ownerShardId = in.readInt32();
                        break;

                    case FIELD_DATA:
                        final int limit = in.readRawVarint32();
                        final int oldLimit = in.pushLimit(limit);
                        var stream = new WriteRequestInputStream(new StockpileDeserializer(in), SerializerVersion.FLAT_WITH_STATE);
                        for (long i = 0; i < recordCount; i++) {
                            stream.readEventTo(archiveByLocalId, projectId, ownerShardId);
                        }
                        in.popLimit(oldLimit);
                        break;

                    default:
                        // skip unknown field
                        in.skipField(tag);
                        break;
                }
            }
        } catch (IOException e) {
            throw new UncheckedIOException("cannot deserialize StockpileShardWriteRequest", e);
        }
    }
}
