package ru.yandex.stockpile.client.writeRequest;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.writeRequest.serializers.SerializerVersion;
import ru.yandex.stockpile.client.writeRequest.serializers.WriteRequestPointDeserializer;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class WriteRequestInputStream {

    private final StockpileDeserializer deserializer;
    private final WriteRequestPointDeserializer pointDeserializer;

    public WriteRequestInputStream(StockpileDeserializer deserializer, SerializerVersion version) {
        this.deserializer = deserializer;
        this.pointDeserializer = version.deserializerGen.get();
    }

    @Nonnull
    private WriteRequestEventType readEventType() {
        int eventTypeByte = deserializer.readByteUnsigned();
        if (eventTypeByte == 0) {
            throw new IllegalStateException("event type byte cannot be 0");
        }
        return WriteRequestEventType.byNumber(eventTypeByte);
    }

    public void readEventTo(Long2ObjectMap<MetricArchiveMutable> target, EProjectId projectId, int stockpileOwnerShardId) {
        WriteRequestEventType eventType = readEventType();
        switch (eventType) {
            case EVENT_POINT:
                readPointTo(target, projectId, stockpileOwnerShardId);
                break;
            case EVENT_DELETE_DATA:
                readDeleteData(target);
                break;
            case EVENT_DELETE_DATA_WITH_TS:
                readDeleteDataWithTs(target);
                break;
            default:
                throw new RuntimeException("Unknown event type: " + eventType);
        }
    }

    private void readPointTo(Long2ObjectMap<MetricArchiveMutable> target, EProjectId projectId, int stockpileOwnerShardId) {
        pointDeserializer.readValueTo(target, deserializer, projectId, stockpileOwnerShardId);
    }

    private void readDeleteData(Long2ObjectMap<MetricArchiveMutable> target) {
        pointDeserializer.readDeleteDataTo(target, deserializer);
    }

    private void readDeleteDataWithTs(Long2ObjectMap<MetricArchiveMutable> target) {
        pointDeserializer.readDeleteDataWithTsTo(target, deserializer);
    }

    public boolean atEof() {
        return deserializer.atEof();
    }
}
