package ru.yandex.stockpile.client.writeRequest.serializers;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;

import ru.yandex.commune.protobuf.WireFormat;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class LogHistogramFieldSerializer implements ProtobufFieldSerializer {
    private final int tag;

    public LogHistogramFieldSerializer(int fieldNumber) {
        tag = WireFormat.makeTag(fieldNumber, com.google.protobuf.WireFormat.WIRETYPE_VARINT);
    }

    @Override
    public int tag() {
        return tag;
    }

    public void writeValue(StockpileSerializer serializer, LogHistogram logHistogram) {
        FieldTagSerializer.writeFieldTag(serializer, tag);
        serializer.writeVarint32(CodedOutputStream.encodeZigZag32(logHistogram.getStartPower()));
        serializer.writeVarint64(logHistogram.getCountZero());
        serializer.writeVarint32(logHistogram.getMaxBucketsSize());
        serializer.writeDouble(logHistogram.getBase());
        serializer.writeVarint32(logHistogram.countBucket());

        for (int index = 0; index < logHistogram.countBucket(); index++) {
            serializer.writeDouble(logHistogram.getBucketValue(index));
        }
    }

    public LogHistogram readValue(int fieldTag, StockpileDeserializer deserializer) {
        if (fieldTag != tag) {
            throw new IllegalStateException(fieldTag + " != " + tag);
        }

        int startPower = CodedInputStream.decodeZigZag32(deserializer.readVarint32());
        long countZero = deserializer.readVarint64();
        int maxBucketSize = deserializer.readVarint32();
        double base = deserializer.readDouble();
        int countBuckets = deserializer.readVarint32();

        double[] buckets = new double[countBuckets];
        for (int index = 0; index < countBuckets; index++) {
            buckets[index] = deserializer.readDouble();
        }

        return LogHistogram.newBuilder()
            .setStartPower(startPower)
            .setCountZero(countZero)
            .setMaxBucketsSize(maxBucketSize)
            .setBase(base)
            .setBuckets(buckets)
            .build();
    }
}
