package ru.yandex.stockpile.client.writeRequest.serializers.flat;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.point.column.LongValueColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.SummaryDoubleColumn;
import ru.yandex.solomon.model.point.column.SummaryInt64Column;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.writeRequest.serializers.FieldTagSerializer;
import ru.yandex.stockpile.client.writeRequest.serializers.WriteRequestPointDeserializer;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class WriteRequestPointFlatDeserializer implements WriteRequestPointDeserializer {
    private FlatEncoderState state = new FlatEncoderState();
    private AggrPoint tempPoint = new AggrPoint();

    @Override
    public void readValueTo(
        Long2ObjectMap<MetricArchiveMutable> target,
        StockpileDeserializer deserializer,
        EProjectId projectId,
        int stockpileOwnerShardId)
    {
        int mask = 0;

        for (;;) {
            int fieldTag = FieldTagSerializer.readFieldTag(deserializer);

            if (fieldTag == FieldTagSerializer.SEPARATOR_TAG) {
                break;
            } else if (fieldTag == AllFields.SIMILAR_FIELDS_MASK_ENCODER.tag) {
                int similarFieldsMask = AllFields.SIMILAR_FIELDS_MASK_ENCODER.readValue(fieldTag, deserializer);
                mask |= similarFieldsMask;
            } else if (FlatLocalIdEncoder.tryDecodeLocalId(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case */
            } else if (FlatValueEncoder.tryDecodePointValue(fieldTag, deserializer, state)) {
                mask |= ValueColumn.mask;
            } else if (FlatTsEncoder.tryDecodePointTsMillis(fieldTag, deserializer, state)) {
                mask |= TsColumn.mask;
            } else if (FlatDecimPolicyIdEncoder.tryDecodeDecimPolicyId(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case  */
            } else if (FlatMetricTypeEncoder.tryDecodeMetricType(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case  */
            } else if (FlatStepEncoder.tryDecodePointStepMillis(fieldTag, deserializer, state)) {
                mask |= StepColumn.mask;
            } else if (FlatMergeEncoder.tryDecodePointMerge(fieldTag, deserializer, state)) {
                mask |= MergeColumn.mask;
            } else if (FlatCountEncoder.tryDecodePointCountValue(fieldTag, deserializer, state)) {
                mask |= CountColumn.mask;
            } else if (FlatLogHistogramEncoder.tryDecodeLogHistogram(fieldTag, deserializer, state)) {
                mask |= LogHistogramColumn.mask;
            } else if (FlatHistogramEncoder.tryDecodeHistogram(fieldTag, deserializer, state)) {
                mask |= HistogramColumn.mask;
            } else if (FlatSummaryInt64Encoder.tryDecodeSummaryInt64(fieldTag, deserializer, state)) {
                mask |= SummaryInt64Column.mask;
            } else if (FlatSummaryDoubleEncoder.tryDecodeSummaryDouble(fieldTag, deserializer, state)) {
                mask |= SummaryDoubleColumn.mask;
            } else if (FlatLongValueEncoder.tryDecodePointValue(fieldTag, deserializer, state)) {
                mask |= LongValueColumn.mask;
            } else {
                deserializer.skipProtobufField(fieldTag);
            }
        }

        if (!DecimPoliciesPredefined.isKnownId(state.lastDecimPolicyId)) {
            throw new IllegalArgumentException("invalid decim policy " + state.lastDecimPolicyId);
        }

        state.copyTo(mask, tempPoint);

        if (mask != 0 && !InstantUtils.isGoodMillis(tempPoint.tsMillis)) {
            throw new IllegalArgumentException(
                "invalid timestamp " + InstantUtils.formatToSeconds(tempPoint.tsMillis) +
                    " in metric " + state.lastStockpileLocalId);
        }

        MetricArchiveMutable archive = archiveRefByLocalId(target, state.lastStockpileLocalId);
        archive.setDecimPolicyId(state.lastDecimPolicyId);
        archive.setOwnerProjectIdEnum(projectId);
        archive.setOwnerShardId(stockpileOwnerShardId);
        archive.setType(state.lastType);
        if (mask != 0) {
            archive.addRecord(tempPoint);
        }
    }

    @Override
    public void readDeleteDataTo(Long2ObjectMap<MetricArchiveMutable> target, StockpileDeserializer deserializer) {
        for (;;) {
            int fieldTag = FieldTagSerializer.readFieldTag(deserializer);

            if (fieldTag == FieldTagSerializer.SEPARATOR_TAG) {
                break;
            } else if (FlatLocalIdEncoder.tryDecodeLocalId(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case */
            } else {
                deserializer.skipProtobufField(fieldTag);
            }
        }
        archiveRefByLocalId(target, state.lastStockpileLocalId).setDeleteBefore(DeleteBeforeField.DELETE_ALL);
    }

    @Override
    public void readDeleteDataWithTsTo(Long2ObjectMap<MetricArchiveMutable> target, StockpileDeserializer deserializer) {
        for (;;) {
            int fieldTag = FieldTagSerializer.readFieldTag(deserializer);

            if (fieldTag == FieldTagSerializer.SEPARATOR_TAG) {
                break;
            } else if (FlatLocalIdEncoder.tryDecodeLocalId(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case */
            } else if (FlatTsEncoder.tryDecodePointTsMillis(fieldTag, deserializer, state)) {
                /* nop - it was stored in state in any case */
            } else {
                deserializer.skipProtobufField(fieldTag);
            }
        }
        archiveRefByLocalId(target, state.lastStockpileLocalId).setDeleteBefore(state.lastTsMillis);
    }

    private MetricArchiveMutable archiveRefByLocalId(Long2ObjectMap<MetricArchiveMutable> archives, long localId) {
        var archive = archives.get(localId);
        if (archive == null) {
            archive = new MetricArchiveMutable();
            archives.put(localId, archive);
        }
        return archive;
    }
}
