package ru.yandex.stockpile.client.writeRequest.serializers.flat;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.writeRequest.serializers.FieldTagSerializer;
import ru.yandex.stockpile.client.writeRequest.serializers.WriteRequestPointSerializer;

/**
 * @author Maksim Leonov (nohttp@)
 * How this works:
 *
 * Serializer has an (initially empty) state.
 *
 * For every field we are going to write to the stream, we only write it to the stream
 * only if it's value is not equal to the value of the corresponding field in the state.
 * We also update the field in the state with this new value.
 *
 * We remember a set of fields we did not write into stream due to this rule in a similarFieldsMask,
 * and at the end of the point we write this mask in the stream too (same rule applies: we do note
 * serialize it if it is equal to the mask stored in the state)
 */
@ParametersAreNonnullByDefault
public class WriteRequestPointFlatSerializer implements WriteRequestPointSerializer {
    private FlatEncoderState state = new FlatEncoderState();

    public WriteRequestPointFlatSerializer() {}

    @Override
    public void writeValue(
        StockpileSerializer serializer,
        long localId,
        int decimPolicyId,
        MetricType type,
        int pointMask,
        AggrPointData point)
    {
        int similarFieldsMask = 0;
        FlatLocalIdEncoder.encodeLocalId(serializer, state, localId);
        FlatDecimPolicyIdEncoder.encodeDecimPolicyId(serializer, state, decimPolicyId);
        FlatMetricTypeEncoder.encodeMetricType(serializer, state, type);

        if ((pointMask & ValueColumn.mask) != 0) {
            similarFieldsMask |= FlatValueEncoder.encodePointValue(serializer, state, point.getValueNum(), point.getValueDenom());
        }
        if ((pointMask & TsColumn.mask) != 0) {
            similarFieldsMask |= FlatTsEncoder.encodePointTsMillisPossiblySame(serializer, state, point.getTsMillis());
        }
        if ((pointMask & MergeColumn.mask) != 0) {
            similarFieldsMask |= FlatMergeEncoder.encodePointMerge(serializer, state, point.isMerge());
        }
        if ((pointMask & CountColumn.mask) != 0) {
            similarFieldsMask |= FlatCountEncoder.encodePointCount(serializer, state, Math.toIntExact(point.getCount()));
        }
        if ((pointMask & StepColumn.mask) != 0) {
            similarFieldsMask |= FlatStepEncoder.encodePointStep(serializer, state, point.getStepMillis());
        }
        if ((pointMask & LogHistogramColumn.mask) != 0) {
            similarFieldsMask |= FlatLogHistogramEncoder.encodeLogHistogram(serializer, state, point.logHistogram);
        }
        if ((pointMask & HistogramColumn.mask) != 0) {
            similarFieldsMask |= FlatHistogramEncoder.encodeHistogram(serializer, state, point.histogram);
        }
        if (HasColumnSet.hasColumn(pointMask, StockpileColumn.ISUMMARY)) {
            similarFieldsMask |= FlatSummaryInt64Encoder.encodeSummaryInt64(serializer, state, point.summaryInt64);
        }
        if (HasColumnSet.hasColumn(pointMask, StockpileColumn.DSUMMARY)) {
            similarFieldsMask |= FlatSummaryDoubleEncoder.encodeSummaryDouble(serializer, state, point.summaryDouble);
        }
        if (StockpileColumn.LONG_VALUE.isInSet(pointMask)) {
            similarFieldsMask |= FlatLongValueEncoder.encodePointValue(serializer, state, point.longValue);
        }

        if (similarFieldsMask != 0) {
            AllFields.SIMILAR_FIELDS_MASK_ENCODER.writeValue(serializer, similarFieldsMask);
        }

        FieldTagSerializer.writeEndTag(serializer);
    }

    @Override
    public void writeDeleteData(StockpileSerializer serializer, long localId) {
        FlatLocalIdEncoder.encodeLocalId(serializer, state, localId);

        FieldTagSerializer.writeEndTag(serializer);
    }

    @Override
    public void writeDeleteDataWithTs(StockpileSerializer serializer, long localId, long tsMillis) {
        FlatLocalIdEncoder.encodeLocalId(serializer, state, localId);
        FlatTsEncoder.encodePointTsMillisPossiblySame(serializer, state, tsMillis);

        FieldTagSerializer.writeEndTag(serializer);
    }
}
