package ru.yandex.stockpile.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.net.HostAndPort;
import com.google.protobuf.Message;
import io.netty.util.ResourceLeakDetector;
import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.mem.ShardCommandAccumulator;
import ru.yandex.stockpile.client.util.ChunkEncoder;
import ru.yandex.stockpile.client.util.Encoder;
import ru.yandex.stockpile.client.util.InMemoryStockpileCluster;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static ru.yandex.stockpile.client.TestUtil.point;
import static ru.yandex.stockpile.client.TestUtil.syncReadOne;
import static ru.yandex.stockpile.client.TestUtil.syncWriteOne;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileClientRetryTest {
    private static final Logger logger = LoggerFactory.getLogger(StockpileClientRetryTest.class);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(2, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    @Rule
    public TestName testName = new TestName();

    private InMemoryStockpileCluster cluster;
    private StockpileClient client;

    @BeforeClass
    public static void beforeClass() {
        ResourceLeakDetector.setLevel(ResourceLeakDetector.Level.ADVANCED);
    }

    @Before
    public void setUp() throws Exception {
        logger.debug("setUp for test: " + testName.getMethodName());
        cluster = InMemoryStockpileCluster.newBuilder()
                .serverCount(3)
                .shardRange(1, 42)
                .inProcess()
                .build();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(500, TimeUnit.MILLISECONDS)
                    .setKeepAliveDelay(100, TimeUnit.MILLISECONDS)
                    .setKeepAliveTimeout(500, TimeUnit.MILLISECONDS)
                    .setChannelFactory(new InProcessChannelFactory()))
                .setExpireClusterMetadata(500, TimeUnit.MILLISECONDS)
                .setMetaDataRequestRetryDelay(10, TimeUnit.MILLISECONDS)
                .setRetryStopStrategy(StopStrategies.neverStop())
                .build();

        client = StockpileClients.create(cluster.getServerList(), options);
        while (client.getReadyShardsCount() < 42) {
            client.forceUpdateClusterMetaData().join();
        }
        logger.debug("Start test: " + testName.getMethodName());
    }

    @After
    public void tearDown() throws Exception {
        if (client != null) {
            client.close();
        }

        if (cluster != null) {
            cluster.stop();
        }
    }

    @Test
    public void retryCreateMetric() throws Exception {
        // shutdown cluster before create metric
        stopCluster();

        CreateMetricRequest request = CreateMetricRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(2)
                    .build())
                .setType(MetricType.DGAUGE)
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(15))
                .build();

        CompletableFuture<CreateMetricResponse> future = client.createMetric(request);
        expectResultNotExistsSometimes(future);

        // cluster not ready
        restartCluster();

        CreateMetricResponse response = future.join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void retryReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);
        syncWriteOne(client, TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build()
        );

        stopNodeOwnMetric(metricId);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                .build();

        CompletableFuture<TReadResponse> future = client.readOne(request);
        expectResultNotExistsSometimes(future);

        restartNodeOwnMetric(metricId);
        TReadResponse response = future.join();

        TReadResponse expected = TReadResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build();

        assertThat(response, equalTo(expected));
    }

    @Test
    public void retryReadCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);
        syncWriteOne(client, TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build()
        );

        stopNodeOwnMetric(metricId);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .build();

        CompletableFuture<TCompressedReadResponse> future = client.readCompressedOne(request);
        expectResultNotExistsSometimes(future);

        restartNodeOwnMetric(metricId);
        TCompressedReadResponse response = future.join();

        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
        List<TPoint> points = ChunkEncoder.decode(Encoder.V42_POINTS_CAPACITY, response.getChunksList());
        assertThat(points, equalTo(Collections.singletonList(point("2013-06-21T15:00:00Z", 1))));
    }

    @Test
    public void retryMultiTreadReadOne() throws Exception {
        Map<MetricId, TReadResponse> expectedReads = new HashMap<>();
        for (int shardId : cluster.getShards()) {
            MetricId metricId = TestUtil.createMetric(client, MetricType.DGAUGE, MetricId.newBuilder().setShardId(shardId).build());

            TWriteRequest writeRequest = TWriteRequest.newBuilder()
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .setMetricId(metricId)
                    .addPoints(point("2013-06-21T15:00:00Z", shardId))
                    .build();

            syncWriteOne(client, writeRequest);

            TReadResponse expect = TReadResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.OK)
                    .setMetricId(metricId)
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addPoints(point("2013-06-21T15:00:00Z", shardId))
                    .build();

            expectedReads.put(metricId, expect);
        }

        HostAndPort nodeToStop = cluster.getServerList().iterator().next();
        cluster.forceStopServer(nodeToStop);

        List<CompletableFuture<TReadResponse>> futures = expectedReads.keySet().parallelStream()
                .map(metricId -> TReadRequest.newBuilder()
                        .setMetricId(metricId)
                        .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                        .build()
                ).map(client::readOne)
                .collect(Collectors.toList());

        cluster.restartServer(nodeToStop);

        Map<MetricId, TReadResponse> metricIdToResponse = futures.stream()
                .map(CompletableFuture::join)
                .collect(Collectors.toMap(TReadResponse::getMetricId, Function.identity()));

        for (Map.Entry<MetricId, TReadResponse> entry : expectedReads.entrySet()) {
            assertThat(metricIdToResponse.get(entry.getKey()), equalTo(entry.getValue()));
        }
    }

    @Test
    public void retryWriteOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        stopNodeOwnMetric(metricId);

        TWriteRequest request = TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build();

        CompletableFuture<TWriteResponse> future = client.writeOne(request);
        expectResultNotExistsSometimes(future);
        restartNodeOwnMetric(metricId);

        TWriteResponse response = future.join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void retryWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);
        stopNodeOwnMetric(metricId);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                .addAllChunks(ChunkEncoder.encode(Encoder.V41_OLDER, Collections.singletonList(
                        point("2013-06-21T15:00:00Z", 1)
                )))
                .build();

        CompletableFuture<TCompressedWriteResponse> future = client.writeCompressedOne(request);
        expectResultNotExistsSometimes(future);
        restartNodeOwnMetric(metricId);

        TCompressedWriteResponse response = future.join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void skipRetryReadOneWhenMetricNotExist() throws Exception {
        MetricId existMetricId = createMetric(MetricType.DGAUGE);
        MetricId notExistsMetricId = existMetricId.toBuilder().setLocalId(123L).build();

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(notExistsMetricId)
                .build();

        TReadResponse response = client.readOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void skipRetryReadCompressedOneWhenMetricNotExist() throws Exception {
        MetricId existMetricId = createMetric(MetricType.DGAUGE);
        MetricId notExistsMetricId = existMetricId.toBuilder().setLocalId(123L).build();

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(notExistsMetricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void skipRetryWriteOneWhenMetricNotExists() throws Exception {
        MetricId existMetricId = createMetric(MetricType.DGAUGE);
        MetricId notExistsMetricId = existMetricId.toBuilder().setLocalId(123L).build();

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(notExistsMetricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void skipRetryWriteCompressedOneWhenMetricNotExists() throws Exception {
        MetricId existMetricId = createMetric(MetricType.DGAUGE);
        MetricId notExistsMetricId = existMetricId.toBuilder().setLocalId(123L).build();

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .setMetricId(notExistsMetricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30))
                .addAllChunks(ChunkEncoder.encode(Encoder.V41_OLDER, Collections.singletonList(
                        point("2013-06-21T15:00:00Z", 1)
                )))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void skipRetryTimeoutReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(System.currentTimeMillis() - 1)
                .build();

        TReadResponse response = client.readOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void skipRetryTimeoutReadCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        restartNodeOwnMetric(metricId);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .setDeadline(System.currentTimeMillis() - 1)
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void skipRetryTimeoutWriteOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(lotsOfPoints)
                .setDeadline(System.currentTimeMillis() - 1)
                .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void skipRetryTimeoutWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        restartNodeOwnMetric(metricId);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(MetricType.DGAUGE)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, lotsOfPoints))
                .setDeadline(System.currentTimeMillis())
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void skipRetryReadCompressedOneWhenUnsupportedBinaryFormat() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion() + 1)
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT));
    }

    @Test
    public void skipRetryWriteCompressedOneWhenUnsupportedBinaryFormat() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion() + 1)
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, Collections.singletonList(
                        point("2013-06-21T15:00:00Z", 2)
                )))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT));
    }

    @Test
    public void skipRetryWriteOneWhenCorruptedData() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, Collections.singletonList(
                        point("2013-06-21T15:00:00Z", 3)
                )))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.CORRUPTED_BINARY_DATA));
    }

    @Test
    public void retryStopWhenDeadlineExceededReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(System.currentTimeMillis() + 100)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TReadResponse> future = client.readOne(request);
        TReadResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDeadlineExceededReadCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .setDeadline(System.currentTimeMillis() + 100)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TCompressedReadResponse> future = client.readCompressedOne(request);
        TCompressedReadResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDeadlineExceededWriteOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(lotsOfPoints)
                .setDeadline(System.currentTimeMillis() + 100)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TWriteResponse> future = client.writeOne(request);
        TWriteResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDeadlineExceededWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(MetricType.DGAUGE)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V41_OLDER, lotsOfPoints))
                .setDeadline(System.currentTimeMillis() + 100)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TCompressedWriteResponse> future = client.writeCompressedOne(request);
        TCompressedWriteResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDeadlineExceededCreateMetric() throws Exception {
        CreateMetricRequest request = CreateMetricRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .setDeadline(System.currentTimeMillis() + 100)
                .build();

        stopCluster();

        CreateMetricResponse response = client.createMetric(request).join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededCreateMetric() throws Exception {
        CreateMetricRequest request = CreateMetricRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .build();

        stopCluster();

        CreateMetricResponse response = client.createMetric(request).join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TReadResponse> future = client.readOne(request);
        TReadResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededReadCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TCompressedReadResponse> future = client.readCompressedOne(request);
        TCompressedReadResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededWriteOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(lotsOfPoints)
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TWriteResponse> future = client.writeOne(request);
        TWriteResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        long pointStartTime = System.currentTimeMillis();
        List<TPoint> lotsOfPoints = new ArrayList<>();
        for (int index = 0; index< 100; index++) {
            TPoint point = TPoint.newBuilder()
                    .setTimestampsMillis(pointStartTime + index * 15000)
                    .setDoubleValue(index)
                    .build();

            lotsOfPoints.add(point);
        }

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(MetricType.DGAUGE)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, lotsOfPoints))
                .build();

        stopNodeOwnMetric(metricId);

        CompletableFuture<TCompressedWriteResponse> future = client.writeCompressedOne(request);
        TCompressedWriteResponse response = future.join();
        assertThat(response.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void retryStopWhenDefaultRequestTimeoutExceededMultiThreadReadOne() throws Exception {
        List<MetricId> metricIds = new ArrayList<>(cluster.getShards().length);
        for (int shardId : cluster.getShards()) {
            MetricId metricId = TestUtil.createMetric(client, MetricType.DGAUGE, MetricId.newBuilder().setShardId(shardId).build());
            metricIds.add(metricId);
        }

        stopCluster();

        metricIds.parallelStream()
                .map(metricId -> TReadRequest.newBuilder()
                        .setMetricId(metricId)
                        .build())
                .map(request -> client.readOne(request)
                        .thenAccept(response -> {
                            assertThat(request.getMetricId().toString(), response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
                        }))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfUnit))
                .join();
    }

    @Test
    public void retryBulkShardWritesAsBinary() throws Exception {
        MetricId metricIdWithoutLocalId = createMetric(MetricType.DGAUGE).toBuilder().clearLocalId().build();

        Map<MetricId, List<TPoint>> metricIdToExpectPoints = IntStream.range(1, 10)
                .mapToObj(ignore -> TestUtil.createMetric(client, MetricType.DGAUGE, metricIdWithoutLocalId))
                .collect(Collectors.toMap(Function.identity(), TestUtil::randomDoublePoints));

        ShardCommandAccumulator accumulator = new ShardCommandAccumulator(metricIdWithoutLocalId.getShardId());

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            accumulator.append(TWriteRequest.newBuilder()
                    .setMetricId(entry.getKey())
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addAllPoints(entry.getValue())
                    .build());
        }

        long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30);
        stopNodeOwnMetric(metricIdWithoutLocalId);
        CompletableFuture<TShardCommandResponse> future = client.bulkShardCommand(accumulator.buildWithDeadline(deadline));
        expectResultNotExistsSometimes(future);
        restartNodeOwnMetric(metricIdWithoutLocalId);

        TShardCommandResponse result = future.join();
        assertThat(result.getStatusMessage(), result.getStatus(), equalTo(EStockpileStatusCode.OK));

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            TReadResponse response = syncReadOne(client, TReadRequest.newBuilder().setMetricId(entry.getKey()).build());

            assertThat(response.getPointsList(), equalTo(entry.getValue()));
        }
    }

    @Test
    public void bulkShardWritesAsProto() throws Exception {
        MetricId metricIdWithoutLocalId = createMetric(MetricType.DGAUGE).toBuilder().clearLocalId().build();

        Map<MetricId, List<TPoint>> metricIdToExpectPoints = IntStream.range(1, 100)
                .mapToObj(ignore -> TestUtil.createMetric(client, MetricType.DGAUGE, metricIdWithoutLocalId))
                .collect(Collectors.toMap(Function.identity(), TestUtil::randomDoublePoints));

        List<TCommandRequest> commands = metricIdToExpectPoints.entrySet().stream()
                .map(entry -> TCommandRequest.newBuilder()
                        .setWrite(TWriteRequest.newBuilder()
                                .setMetricId(entry.getKey())
                                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                                .addAllPoints(entry.getValue())
                                .build())
                        .build())
                .collect(Collectors.toList());

        TShardCommandRequest request = TShardCommandRequest.newBuilder()
                .setShardId(metricIdWithoutLocalId.getShardId())
                .addAllCommands(commands)
                .setDeadline(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60))
                .build();

        stopNodeOwnMetric(metricIdWithoutLocalId);
        CompletableFuture<TShardCommandResponse> future = client.bulkShardCommand(request);
        expectResultNotExistsSometimes(future);
        restartNodeOwnMetric(metricIdWithoutLocalId);

        TShardCommandResponse result = future.join();
        assertThat(result.getStatusMessage(), result.getStatus(), equalTo(EStockpileStatusCode.OK));

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            TReadResponse response = syncReadOne(client, TReadRequest.newBuilder().setMetricId(entry.getKey()).build());

            assertThat(response.getPointsList(), equalTo(entry.getValue()));
        }
    }

    private <Response extends Message> void expectResultNotExistsSometimes(CompletableFuture<Response> future)
            throws ExecutionException, InterruptedException {
        try {
            Response response = future.get(100, TimeUnit.MILLISECONDS);
            fail("Cluster currently in shutdown state, client should retry creating until cluster " +
                    "will be not ready. Response: " + response
            );
        } catch (TimeoutException timeout) {
            // ok, cluster unavailable
        }
    }

    private void stopNodeOwnMetric(MetricId metricId) throws InterruptedException {
        HostAndPort nodeWithData = cluster.getServerWithShard(metricId.getShardId());
        cluster.forceStopServer(nodeWithData);
    }

    private void restartNodeOwnMetric(MetricId metricId) throws IOException, InterruptedException {
        HostAndPort nodeWithData = cluster.getServerWithShard(metricId.getShardId());
        cluster.restartServer(nodeWithData);
    }

    private void stopCluster() throws InterruptedException {
        for (HostAndPort address : cluster.getServerList()) {
            cluster.forceStopServer(address);
        }
    }

    private void restartCluster() throws IOException, InterruptedException {
        for (HostAndPort address : cluster.getServerList()) {
            cluster.restartServer(address);
        }
    }

    private MetricId createMetric(MetricType type) {
        int shardId = cluster.getShards()[ThreadLocalRandom.current().nextInt(cluster.getShards().length)];
        return TestUtil.createMetric(client, type, MetricId.newBuilder()
            .setShardId(shardId)
            .build());
    }
}
