package ru.yandex.stockpile.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Range;
import com.google.common.net.HostAndPort;
import io.netty.util.ResourceLeakDetector;
import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.CreateMetricResponse;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TTimeseriesUncompressed;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.client.mem.AccumulatedShardCommand;
import ru.yandex.stockpile.client.mem.ShardCommandAccumulator;
import ru.yandex.stockpile.client.util.ChunkEncoder;
import ru.yandex.stockpile.client.util.Encoder;
import ru.yandex.stockpile.client.util.InMemoryStockpileCluster;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.stockpile.client.TestUtil.metricId;
import static ru.yandex.stockpile.client.TestUtil.point;
import static ru.yandex.stockpile.client.TestUtil.syncCompressedReadOne;
import static ru.yandex.stockpile.client.TestUtil.syncReadOne;
import static ru.yandex.stockpile.client.TestUtil.syncWriteCompressedOne;
import static ru.yandex.stockpile.client.TestUtil.syncWriteOne;
import static ru.yandex.stockpile.client.TestUtil.timeToMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileClientTest {
    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    private InMemoryStockpileCluster cluster;
    private StockpileClient client;

    @BeforeClass
    public static void beforeClass() {
        ResourceLeakDetector.setLevel(ResourceLeakDetector.Level.ADVANCED);
    }

    @Before
    public void setUp() throws Exception {
        cluster = InMemoryStockpileCluster.newBuilder()
                .serverCount(4)
                .shardRange(1, 42)
                .inProcess()
                .build();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setChannelFactory(new InProcessChannelFactory()))
                .setExpireClusterMetadata(1, TimeUnit.SECONDS)
                .build();

        client = StockpileClients.create(cluster.getServerList(), options);
        while (client.getReadyShardsCount() < 42) {
            client.forceUpdateClusterMetaData().join();
        }
    }

    @After
    public void tearDown() throws Exception {
        if (client != null) {
            client.close();
        }

        if (cluster != null) {
            cluster.stop();
        }
    }

    @Test
    public void notExistsMetricWriteOne() throws Exception {
        MetricId metricId = metricId(8, 123L);

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-21T15:54:26Z", 3))
                .addPoints(point("2017-06-21T15:54:30Z", 2))
                .addPoints(point("2017-06-21T15:54:50Z", 1))
                .build();

        TWriteResponse response = client.writeOne(request).get();
        assertThat(response.toString(), response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void notExistsMetricWriteCompressedOne() throws Exception {
        MetricId metricId = metricId(8, 123L);
        List<TPoint> points = Arrays.asList(
                point("2017-06-21T15:54:26Z", 3),
                point("2017-06-21T15:54:30Z", 2),
                point("2017-06-21T15:54:50Z", 1)
        );

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, points))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).get();
        assertThat(response.toString(), response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void notExistsMetricReadOne() throws Exception {
        MetricId metricId = metricId(10, 100500L);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build();

        TReadResponse response = client.readOne(request).get();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void notExistsMetricReadCompressedOne() throws Exception {
        MetricId metricId = metricId(10, 321L);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).get();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void createAndWriteOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-21T15:54:26Z", 3))
                .addPoints(point("2017-06-21T15:54:30Z", 2))
                .addPoints(point("2017-06-21T15:54:50Z", 1))
                .build();

        TWriteResponse response = client.writeOne(request).get();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void createAndWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);
        List<TPoint> points = Arrays.asList(
                point("2017-06-21T15:00:00Z", 1),
                point("2017-06-21T16:00:00Z", 2),
                point("2017-06-21T17:00:00Z", 3)
        );

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, points))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).get();
        assertThat(response.toString(), response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void createAndReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build();

        TReadResponse response = client.readOne(request).get();

        TReadResponse expect = TReadResponse.newBuilder()
                .setMetricId(metricId)
                .setStatus(EStockpileStatusCode.OK)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .build();

        assertThat(response, equalTo(expect));
    }

    @Test
    public void createAndCompressedReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        int version = client.getCompatibleCompressFormat().lowerEndpoint();

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(version)
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).get();

        TCompressedReadResponse expect = TCompressedReadResponse.newBuilder()
                .setMetricId(metricId)
                .setStatus(EStockpileStatusCode.OK)
                .setType(MetricType.DGAUGE)
                .setBinaryVersion(version)
                .build();

        assertThat(response, equalTo(expect));
    }

    @Test
    public void writeOneAndReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
                point("2017-06-21T15:54:26Z", 3),
                point("2017-06-21T15:54:30Z", 2),
                point("2017-06-21T15:54:50Z", 1)
        );

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(expectPoints)
                .build()
        );

        TReadResponse readResponse = client.readOne(TReadRequest.newBuilder().setMetricId(metricId).build()).get();

        TReadResponse expected = TReadResponse.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(expectPoints)
                .setStatus(EStockpileStatusCode.OK)
                .build();

        assertThat(readResponse, equalTo(expected));
    }

    @Test
    public void writeOneAndReadCompressedOneNew() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
                point("2017-06-21T15:54:26Z", 5),
                point("2017-06-21T15:54:30Z", 1),
                point("2017-06-21T15:54:50Z", 3)
        );

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(expectPoints)
                .build()
        );

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .build();

        TCompressedReadResponse response = syncCompressedReadOne(client, request);
        List<TPoint> resultPoints = ChunkEncoder.decode(Encoder.V42_POINTS_CAPACITY, response.getChunksList());
        assertThat(resultPoints, equalTo(expectPoints));
    }

    @Test
    public void writeOneAndReadCompressedOneOld() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
                point("2017-06-21T15:54:26Z", 5),
                point("2017-06-21T15:54:30Z", 4),
                point("2017-06-21T15:54:50Z", 3)
        );

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addAllPoints(expectPoints)
                .build()
        );

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .build();

        TCompressedReadResponse response = syncCompressedReadOne(client, request);
        List<TPoint> resultPoints = ChunkEncoder.decode(Encoder.V41_OLDER, response.getChunksList());
        assertThat(resultPoints, equalTo(expectPoints));
    }

    @Test
    public void writeCompressedOneNewAndReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
                point("2017-06-21T15:54:26Z", 9),
                point("2017-06-21T15:54:30Z", 8),
                point("2017-06-21T15:54:50Z", 7)
        );

        syncWriteCompressedOne(client, TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, expectPoints))
                .build()
        );

        TReadResponse response = syncReadOne(client, TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build()
        );

        assertThat(response.getPointsList(), equalTo(expectPoints));
    }

    @Test
    public void writeCompressedOneOldAndReadOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
                point("2017-06-21T15:54:26Z", 6),
                point("2017-06-21T15:54:30Z", 5),
                point("2017-06-21T15:54:50Z", 4)
        );

        syncWriteCompressedOne(client, TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V41_OLDER, expectPoints))
                .build()
        );

        TReadResponse response = syncReadOne(client, TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build()
        );

        assertThat(response.getPointsList(), equalTo(expectPoints));
    }

    @Test
    public void notAbleReadTimeSeriesFromAnotherShard() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-22T11:00:00Z", 123))
                .build()
        );

        int anotherShardId = IntStream.of(cluster.getShards())
                .filter(shardId -> shardId != metricId.getShardId())
                .findFirst()
                .orElseThrow();

        TReadRequest readRequest = TReadRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                        .setLocalId(metricId.getLocalId())
                        .setShardId(anotherShardId)
                        .build())
                .build();

        TReadResponse readResponse;
        do {
            readResponse = client.readOne(readRequest).get();
        } while (readResponse.getStatus() == EStockpileStatusCode.SHARD_NOT_READY);
        assertThat(readResponse.getStatus(), CoreMatchers.equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void createTimeSeriesWithPredefineShard() throws Exception {
        MetricId metricId = TestUtil.createMetric(client, MetricType.DGAUGE, MetricId.newBuilder().setShardId(8).build());
        assertThat(metricId.getShardId(), CoreMatchers.equalTo(8));
    }

    @Test
    public void createTimeSeriesWithPredefineMetricId() throws Exception {
        MetricId expected = MetricId.newBuilder()
                .setShardId(10)
                .setLocalId(123)
                .build();

        MetricId metricId = TestUtil.createMetric(client, MetricType.DGAUGE, expected);
        Assert.assertThat(metricId, CoreMatchers.equalTo(expected));
    }

    @Test
    public void notAbleCreateMetricWithSameMetricIdTwice() throws Exception {
        MetricId notUniqueMetricId = MetricId.newBuilder()
                .setShardId(10)
                .setLocalId(123)
                .build();

        CreateMetricRequest request =
                CreateMetricRequest.newBuilder()
                        .setType(MetricType.DGAUGE)
                        .setMetricId(notUniqueMetricId)
                        .build();

        assertThat(client.createMetric(request).get().getStatus(), equalTo(EStockpileStatusCode.OK));
        assertThat(client.createMetric(request).get().getStatus(), equalTo(EStockpileStatusCode.METRIC_ALREADY_EXISTS));
    }

    @Test
    public void createMetricConsiderShardLoad() throws Exception {
        int totalShardCount = cluster.getShards().length;

        Map<Integer, Long> shardToCountMetrics =
                IntStream.of(cluster.getShards()).boxed()
                        .collect(Collectors.toMap(Function.identity(), shardId -> 0L));

        for (int index = 0; index < totalShardCount; index++) {
            MetricId metricId = createMetric(MetricType.DGAUGE);

            shardToCountMetrics.compute(metricId.getShardId(), (integer, count) -> count + 1);

            syncWriteOne(client, TWriteRequest.newBuilder()
                    .setMetricId(metricId)
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addPoints(point("2017-06-21T15:54:26Z", 3))
                    .addPoints(point("2017-06-21T15:55:00Z", 2))
                    .addPoints(point("2017-06-21T15:56:00Z", 1))
                    .build()
            );

            client.forceUpdateClusterMetaData().get();
        }

        List<Integer> shardsWithoutMetrics = shardToCountMetrics.entrySet()
                .stream()
                .filter(entry -> entry.getValue() == 0)
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());

        assertTrue(shardsWithoutMetrics.size() < totalShardCount / 2);
    }

    @Test
    public void readMetricWithPredefineTimeRange() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2017-06-21T15:00:00Z", 1))
                .addPoints(point("2017-06-21T15:05:00Z", 2))
                .addPoints(point("2017-06-21T15:10:00Z", 3))
                .addPoints(point("2017-06-21T15:15:00Z", 4))
                .addPoints(point("2017-06-21T15:20:00Z", 5))
                .build()
        );

        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setFromMillis(timeToMillis("2017-06-21T15:05:00Z"))
                .setToMillis(timeToMillis("2017-06-21T15:20:00Z"))
                .build();

        TReadResponse response = client.readOne(request).join();

        TReadResponse expect = TReadResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-21T15:05:00Z", 2))
                .addPoints(point("2017-06-21T15:10:00Z", 3))
                .addPoints(point("2017-06-21T15:15:00Z", 4))
                .build();

        assertThat(response, equalTo(expect));
    }

    @Test
    public void notAbleDeleteMetricFromNotReadyShard() throws Exception {
        DeleteMetricRequest request = DeleteMetricRequest.newBuilder()
                .setMetricId(metricId(50, 123L))
                .setDeadline(System.currentTimeMillis() + TimeUnit.MILLISECONDS.toMillis(100))
                .build();

        DeleteMetricResponse result = client.deleteMetric(request).join();
        assertThat(result.getStatus(), equalTo(EStockpileStatusCode.SHARD_NOT_READY));
    }

    @Test
    public void notAbleDeleteMetricWhenNotExists() throws Exception {
        MetricId existsMetric = createMetric(MetricType.DGAUGE);
        DeleteMetricRequest request = DeleteMetricRequest.newBuilder()
                .setMetricId(metricId(existsMetric.getShardId(), existsMetric.getLocalId() + 1))
                .build();

        DeleteMetricResponse result = client.deleteMetric(request).join();
        assertThat(result.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void correctDeleteMetric() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-21T15:00:00Z", 1))
                .addPoints(point("2017-06-21T15:05:00Z", 2))
                .addPoints(point("2017-06-21T15:10:00Z", 3))
                .build()
        );

        DeleteMetricRequest deleteRequest = DeleteMetricRequest.newBuilder().setMetricId(metricId).build();
        DeleteMetricResponse deleteResponse = client.deleteMetric(deleteRequest).join();
        assertThat(deleteResponse.getStatus(), equalTo(EStockpileStatusCode.OK));

        TReadRequest readRequest = TReadRequest.newBuilder().setMetricId(metricId).build();
        TReadResponse readResponse = client.readOne(readRequest).join();
        assertThat(readResponse.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void notAbleDeleteDataWhenShardNotInit() throws Exception {
        DeleteMetricDataRequest deleteRequest = DeleteMetricDataRequest.newBuilder()
                .setMetricId(metricId(55, 123L))
                .setFromMillis(timeToMillis("2013-01-01T00:00:00Z"))
                .setToMillis(timeToMillis("2017-01-01T00:00:00Z"))
                .build();

        DeleteMetricDataResponse result = client.deleteMetricData(deleteRequest).join();
        assertThat(result.getStatus(), equalTo(EStockpileStatusCode.SHARD_NOT_READY));
    }

    @Test
    public void notAbleDeleteMetricDataWhenNotExists() throws Exception {
        MetricId existsMetric = createMetric(MetricType.DGAUGE);

        DeleteMetricDataRequest deleteRequest = DeleteMetricDataRequest.newBuilder()
                .setMetricId(metricId(existsMetric.getShardId(), existsMetric.getLocalId() + 1))
                .setFromMillis(timeToMillis("2015-01-01T00:00:00Z"))
                .setToMillis(timeToMillis("2017-01-01T00:00:00Z"))
                .build();

        DeleteMetricDataResponse result = client.deleteMetricData(deleteRequest).join();
        assertThat(result.getStatus(), equalTo(EStockpileStatusCode.METRIC_NOT_FOUND));
    }

    @Test
    public void deleteRangeOfDataFromMetric() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .addPoints(point("2013-06-21T15:05:00Z", 2))
                .addPoints(point("2015-06-21T15:10:00Z", 3))
                .addPoints(point("2017-06-21T15:15:00Z", 4))
                .addPoints(point("2017-06-21T15:20:00Z", 5))
                .build()
        );

        DeleteMetricDataRequest deleteRequest = DeleteMetricDataRequest.newBuilder()
                .setMetricId(metricId)
                .setFromMillis(timeToMillis("2013-01-01T00:00:00Z"))
                .setToMillis(timeToMillis("2017-01-01T00:00:00Z"))
                .build();

        DeleteMetricDataResponse deleteResponse = client.deleteMetricData(deleteRequest).join();
        assertThat(deleteResponse.getStatus(), equalTo(EStockpileStatusCode.OK));

        TReadRequest readRequest = TReadRequest.newBuilder().setMetricId(metricId).build();
        TReadResponse readResponse = client.readOne(readRequest).join();

        TReadResponse expectReadResponse = TReadResponse.newBuilder()
                .setMetricId(metricId)
                .setStatus(EStockpileStatusCode.OK)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-06-21T15:15:00Z", 4))
                .addPoints(point("2017-06-21T15:20:00Z", 5))
                .build();

        assertThat(readResponse, equalTo(expectReadResponse));
    }

    @Test
    public void notAnswerNodeNotAffectClusterUpdateState() throws Exception {
        this.client.close();

        HostAndPort notAnswerHost = HostAndPort.fromString("ya.ru:8080");
        List<HostAndPort> serverList = new ArrayList<>(cluster.getServerList());
        serverList.add(notAnswerHost);

        var opt = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setChannelFactory(new InProcessChannelFactory()))
                .setMetaDataRequestTimeOut(300, TimeUnit.MILLISECONDS)
                .setExpireClusterMetadata(2, TimeUnit.MINUTES)
                .build();

        client = StockpileClients.create(serverList, opt);
        client.forceUpdateClusterMetaData().join();

        MetricId metricId = createMetric(MetricType.DGAUGE);

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build()
        );

        client.forceUpdateClusterMetaData().join();

        TReadResponse response = client.readOne(TReadRequest.newBuilder().setMetricId(metricId).build()).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void nodeIncludesToClusterAfterFailWhenWillBeReady() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);
        syncWriteOne(client, TWriteRequest.newBuilder()
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setMetricId(metricId)
                .addPoints(point("2013-06-21T15:00:00Z", 1))
                .build()
        );

        this.client.close();

        var opts = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                .setRequestTimeOut(5, TimeUnit.MINUTES)
                .setChannelFactory(new InProcessChannelFactory()))
                .setExpireClusterMetadata(1, TimeUnit.MINUTES)
                .setMetaDataRequestTimeOut(200, TimeUnit.MILLISECONDS)
                .build();

        this.client = StockpileClients.create(cluster.getServerList(), opts);
        this.client.forceUpdateClusterMetaData().join();

        HostAndPort node = cluster.getServerWithShard(metricId.getShardId());
        cluster.forceStopServer(node);
        this.client.forceUpdateClusterMetaData().join();
        cluster.restartServer(node);

        // Right after server start client some time reject requests with error:
        // WARNING: An exceptionCaught() event was fired, and it reached at the tail of the pipeline.
        // It usually means the last handler in the pipeline did not handle the exception.
        // io.netty.channel.AbstractChannel$AnnotatedConnectException: Connection refused: localhost/127.0.0.1:50003
        while (true) {
            this.client.forceUpdateClusterMetaData().join();
            TReadRequest request = TReadRequest.newBuilder().setMetricId(metricId).build();
            TReadResponse response = client.readOne(request).join();

            if (response.getStatus() == EStockpileStatusCode.OK) {
                break;
            }
        }
    }

    @Test
    public void createMetricExpiredDeadline() throws Exception {
        final long deadline = System.currentTimeMillis();
        CreateMetricRequest request = CreateMetricRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .setMetricId(MetricId.newBuilder()
                    .setShardId(4)
                    .build())
                .setDeadline(deadline)
                .build();

        CreateMetricResponse response = client.createMetric(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void readOneExpiredDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis();
        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(deadline)
                .build();

        TReadResponse response = client.readOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void readCompressedOneExpiredDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() - 1;
        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(client.getCompatibleCompressFormat().upperEndpoint())
                .setDeadline(deadline)
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void writeOneExpiredDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis();
        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setDeadline(deadline)
                .addPoints(point("2017-06-21T15:54:26Z", 3))
                .addPoints(point("2017-06-21T15:54:30Z", 2))
                .addPoints(point("2017-06-21T15:54:50Z", 1))
                .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void writeCompressedOneExpiredDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() - 1;
        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(MetricType.DGAUGE)
                .setDeadline(deadline)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, Arrays.asList(
                        point("2017-06-21T15:54:26Z", 3),
                        point("2017-06-21T15:54:30Z", 2),
                        point("2017-06-21T15:54:50Z", 1)
                )))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
    }

    @Test
    public void createMetricWithCustomDeadline() {
        final long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60);
        CreateMetricRequest request = CreateMetricRequest.newBuilder()
                .setType(MetricType.DGAUGE)
                .setMetricId(MetricId.newBuilder()
                    .setShardId(5)
                    .build())
                .setDeadline(deadline)
                .build();

        CreateMetricResponse response = client.createMetric(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void readOneWithCustomDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60);
        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(deadline)
                .build();

        TReadResponse response = client.readOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void readCompressedOneWithCustomDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60);
        TReadRequest request = TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setDeadline(deadline)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .build();

        TCompressedReadResponse response = client.readCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void writeOneWithCustomDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60);
        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .setDeadline(deadline)
                .addPoints(point("2017-06-21T15:54:26Z", 3))
                .addPoints(point("2017-06-21T15:54:30Z", 2))
                .addPoints(point("2017-06-21T15:54:50Z", 1))
                .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void writeCompressedOneWithCustomDeadline() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        final long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60);
        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(MetricType.DGAUGE)
                .setDeadline(deadline)
                .setBinaryVersion(Encoder.V42_POINTS_CAPACITY.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, Arrays.asList(
                        point("2017-06-21T15:54:26Z", 3),
                        point("2017-06-21T15:54:30Z", 2),
                        point("2017-06-21T15:54:50Z", 1)
                )))
                .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void supportTargetCompressFormat() throws Exception {
        Range<Integer> range = client.getCompatibleCompressFormat();
        assertTrue(range.contains(Encoder.V42_POINTS_CAPACITY.getVersion()));
        assertTrue(range.contains(Encoder.V41_OLDER.getVersion()));
    }

    @Test
    public void corruptedWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> points = Arrays.asList(
                point("2017-06-21T15:54:26Z", 9),
                point("2017-06-21T15:54:30Z", 8),
                point("2017-06-21T15:54:50Z", 7)
        );

        TCompressedWriteResponse response = client.writeCompressedOne(TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(Encoder.V41_OLDER.getVersion())
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, points))
                .build()).join();

        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.CORRUPTED_BINARY_DATA));
    }

    @Test
    public void unsupportedBinaryFormatWriteCompressedOne() throws Exception {
        MetricId metricId = createMetric(MetricType.DGAUGE);

        List<TPoint> points = Arrays.asList(
                point("2017-06-21T15:54:26Z", 9),
                point("2017-06-21T15:54:30Z", 8),
                point("2017-06-21T15:54:50Z", 7)
        );

        TCompressedWriteResponse response = client.writeCompressedOne(TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setBinaryVersion(50)
                .addAllChunks(ChunkEncoder.encode(Encoder.V42_POINTS_CAPACITY, points))
                .build()).join();

        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT));
    }

    @Test
    public void bulkShardWritesAsBinary() throws Exception {
        MetricId metricIdWithoutLocalId = createMetric(MetricType.DGAUGE).toBuilder().clearLocalId().build();

        Map<MetricId, List<TPoint>> metricIdToExpectPoints = IntStream.range(1, 15)
                .mapToObj(ignore -> TestUtil.createMetric(client, MetricType.DGAUGE, metricIdWithoutLocalId))
                .collect(Collectors.toMap(Function.identity(), TestUtil::randomDoublePoints));

        ShardCommandAccumulator accumulator = new ShardCommandAccumulator(metricIdWithoutLocalId.getShardId());

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            accumulator.append(TWriteRequest.newBuilder()
                    .setMetricId(entry.getKey())
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addAllPoints(entry.getValue())
                    .build());
        }

        AccumulatedShardCommand command = accumulator.build();
        TShardCommandResponse result = client.bulkShardCommand(command).join();
        assertThat(result.getStatusMessage(), result.getStatus(), equalTo(EStockpileStatusCode.OK));

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            TReadResponse response = syncReadOne(client, TReadRequest.newBuilder().setMetricId(entry.getKey()).build());

            assertThat(response.getPointsList(), equalTo(entry.getValue()));
        }
    }

    @Test
    public void bulkShardWritesAsProto() throws Exception {
        MetricId metricIdWithoutLocalId = createMetric(MetricType.DGAUGE).toBuilder().clearLocalId().build();

        Map<MetricId, List<TPoint>> metricIdToExpectPoints = IntStream.range(1, 100)
                .mapToObj(ignore -> TestUtil.createMetric(client, MetricType.DGAUGE, metricIdWithoutLocalId))
                .collect(Collectors.toMap(Function.identity(), TestUtil::randomDoublePoints));

        List<TCommandRequest> commands = metricIdToExpectPoints.entrySet().stream()
                .map(entry -> TCommandRequest.newBuilder()
                        .setWrite(TWriteRequest.newBuilder()
                                .setMetricId(entry.getKey())
                                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                                .addAllPoints(entry.getValue())
                                .build())
                        .build())
                .collect(Collectors.toList());

        TShardCommandRequest request = TShardCommandRequest.newBuilder()
                .setShardId(metricIdWithoutLocalId.getShardId())
                .addAllCommands(commands)
                .build();

        TShardCommandResponse result = client.bulkShardCommand(request).join();
        assertThat(result.getStatusMessage(), result.getStatus(), equalTo(EStockpileStatusCode.OK));

        for (Map.Entry<MetricId, List<TPoint>> entry : metricIdToExpectPoints.entrySet()) {
            TReadResponse response = syncReadOne(client, TReadRequest.newBuilder().setMetricId(entry.getKey()).build());

            assertThat(response.getPointsList(), equalTo(entry.getValue()));
        }
    }

    @Test
    public void readUncompressedMany() {
        MetricId alice = createMetric(MetricType.DGAUGE);
        MetricId bob = TestUtil.createMetric(client, MetricType.DGAUGE, alice.toBuilder().clearLocalId().build());
        MetricId eva = TestUtil.createMetric(client, MetricType.DGAUGE, alice.toBuilder().clearLocalId().build());

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(alice)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2018-07-11T14:18:00Z", 1))
                .addPoints(point("2018-07-11T14:18:15Z", 2))
                .addPoints(point("2018-07-11T14:18:30Z", 3))
                .build());

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(bob)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2018-07-11T14:18:00Z", 3))
                .addPoints(point("2018-07-11T14:18:15Z", 2))
                .addPoints(point("2018-07-11T14:18:30Z", 8))
                .build());

        syncWriteOne(client, TWriteRequest.newBuilder()
                .setMetricId(eva)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2018-07-11T14:18:00Z", 5))
                .addPoints(point("2018-07-11T14:18:15Z", 5))
                .addPoints(point("2018-07-11T14:18:30Z", 5))
                .build());

        TUncompressedReadManyResponse response = client.readUncompressedMany(TReadManyRequest.newBuilder()
                .setShardId(alice.getShardId())
                .addLocalIds(alice.getLocalId())
                .addLocalIds(bob.getLocalId())
                .setFromMillis(timeToMillis("2018-07-11T14:18:00Z"))
                .setToMillis(timeToMillis("2018-07-11T14:19:00Z"))
                .build())
                .join();

        MetricData expectedAlice = MetricData.newBuilder()
                .setShardId(alice.getShardId())
                .setLocalId(alice.getLocalId())
                .setUncompressed(TTimeseriesUncompressed.newBuilder()
                        .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                        .addPoints(point("2018-07-11T14:18:00Z", 1))
                        .addPoints(point("2018-07-11T14:18:15Z", 2))
                        .addPoints(point("2018-07-11T14:18:30Z", 3))
                        .build())
                .build();

        MetricData expectedBob = MetricData.newBuilder()
                .setShardId(bob.getShardId())
                .setLocalId(bob.getLocalId())
                .setUncompressed(TTimeseriesUncompressed.newBuilder()
                        .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                        .addPoints(point("2018-07-11T14:18:00Z", 3))
                        .addPoints(point("2018-07-11T14:18:15Z", 2))
                        .addPoints(point("2018-07-11T14:18:30Z", 8))
                        .build())
                .build();

        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.OK));
        assertThat(response.getMetricsList(), iterableWithSize(2));
        assertThat(response.getMetricsList().get(0), equalTo(expectedAlice));
        assertThat(response.getMetricsList().get(1), equalTo(expectedBob));
    }

    @Test
    public void readCompressedMany() {
        MetricId alice = createMetric(MetricType.DGAUGE);
        MetricId bob = TestUtil.createMetric(client, MetricType.DGAUGE, alice.toBuilder().clearLocalId().build());
        MetricId eva = TestUtil.createMetric(client, MetricType.DGAUGE, alice.toBuilder().clearLocalId().build());

        int version = client.getCompatibleCompressFormat().lowerEndpoint();
        TCompressedReadManyResponse response = client.readCompressedMany(TReadManyRequest.newBuilder()
                .setShardId(alice.getShardId())
                .addLocalIds(alice.getLocalId())
                .addLocalIds(bob.getLocalId())
                .setFromMillis(timeToMillis("2018-07-11T14:18:00Z"))
                .setToMillis(timeToMillis("2018-07-11T14:19:00Z"))
                .setBinaryVersion(version)
                .build())
                .join();

        MetricData expectedAlice = MetricData.newBuilder()
                .setShardId(alice.getShardId())
                .setLocalId(alice.getLocalId())
                .setCompressed(TimeSeries.getDefaultInstance())
                .build();

        MetricData expectedBob = MetricData.newBuilder()
                .setShardId(bob.getShardId())
                .setLocalId(bob.getLocalId())
                .setCompressed(TimeSeries.getDefaultInstance())
                .build();

        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.OK));
        assertThat(response.getMetricsList(), iterableWithSize(2));
        assertThat(response.getMetricsList().get(0), equalTo(expectedAlice));
        assertThat(response.getMetricsList().get(1), equalTo(expectedBob));
    }

    private MetricId createMetric(MetricType type) {
        int shardId = cluster.getShards()[ThreadLocalRandom.current().nextInt(cluster.getShards().length)];
        return TestUtil.createMetric(client, type, MetricId.newBuilder()
            .setShardId(shardId)
            .build());
    }
}
