package ru.yandex.stockpile.client.mem;

import java.util.Collection;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.client.ColumnFlagMask;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.stockpile.client.TestUtil.metricId;
import static ru.yandex.stockpile.client.TestUtil.point;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossShardCommandAccumulatorTest {
    private CrossShardCommandAccumulator accumulator = new CrossShardCommandAccumulator();

    @Before
    public void setUp() throws Exception {
        accumulator = new CrossShardCommandAccumulator();
    }

    @After
    public void tearDown() throws Exception {
        accumulator.close();
    }

    @Test
    public void emptyAccumulatorUseZeroBytes() throws Exception {
        assertThat(accumulator.countBytes(), equalTo(0L));
    }

    @Test
    public void afterBuildAccumulatorUseZeroBytes() throws Exception {
        accumulator.append(TWriteRequest.newBuilder()
                .setMetricId(metricId(1, 1111L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 3))
                .addPoints(point("2017-07-12T13:20:00Z", 2))
                .addPoints(point("2017-07-12T13:10:00Z", 1))
                .build());

        Collection<AccumulatedShardCommand> result = null;
        try {
            result = accumulator.build().values();
            assertThat(accumulator.countBytes(), equalTo(0L));
        } finally {
            if (result != null) {
                result.forEach(AccumulatedShardCommand::close);
            }
        }
    }

    @Test
    public void countBytes() throws Exception {
        accumulator.append(TWriteRequest.newBuilder()
                .setMetricId(metricId(1, 1111L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 3))
                .addPoints(point("2017-07-12T13:20:00Z", 2))
                .addPoints(point("2017-07-12T13:10:00Z", 1))
                .build());

        long sizeAfterFirstWrite = accumulator.countBytes();
        assertThat(sizeAfterFirstWrite, not(equalTo(0L)));

        accumulator.markWriteIndex();

        accumulator.append(TWriteRequest.newBuilder()
                .setMetricId(metricId(2, 222L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 3))
                .addPoints(point("2017-07-12T13:10:00Z", 1))
                .build());

        assertThat(sizeAfterFirstWrite, not(equalTo(accumulator.countBytes())));
        accumulator.resetWriterIndex();
        assertThat(sizeAfterFirstWrite, equalTo(accumulator.countBytes()));
    }
}
