package ru.yandex.stockpile.client.mem;

import java.io.IOException;
import java.nio.ReadOnlyBufferException;
import java.util.concurrent.ThreadLocalRandom;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBufInputStream;
import org.junit.After;
import org.junit.Test;

import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.client.ColumnFlagMask;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static ru.yandex.stockpile.client.TestUtil.metricId;
import static ru.yandex.stockpile.client.TestUtil.point;
import static ru.yandex.stockpile.client.TestUtil.timeToMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardCommandAccumulatorTest {
    private ShardCommandAccumulator accumulator;

    @After
    public void tearDown() throws Exception {
        if (accumulator != null) {
            accumulator.close();
        }
    }

    @Test
    public void write() throws Exception {
        int shardId = 123;

        TWriteRequest writeOne = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 1111L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 3))
                .addPoints(point("2017-07-12T13:20:00Z", 2))
                .addPoints(point("2017-07-12T13:10:00Z", 1))
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(writeOne);
        TShardCommandRequest result = deserialize(accumulator.build());

        TShardCommandRequest expected = TShardCommandRequest.newBuilder()
                .setShardId(shardId)
                .addCommands(TCommandRequest.newBuilder().setWrite(writeOne).build())
                .build();

        assertThat(result, equalTo(expected));
    }

    @Test(expected = IllegalArgumentException.class)
    public void writeNotValidShardId() throws Exception {
        int shardId = 111;

        TWriteRequest writeOne = TWriteRequest.newBuilder()
                .setMetricId(metricId(333, 222L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 1213))
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(writeOne);
        fail("Batch builder build request for single shard id and should restrict add request with another one");
    }

    @Test(expected = IllegalStateException.class)
    public void writeNotAbleAfterBuild() throws Exception {
        int shardId = 111;

        TWriteRequest writeOne = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 222L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 1213))
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(writeOne);
        try (AccumulatedShardCommand commands = accumulator.build()) {
            accumulator.append(writeOne);
        }
    }

    @Test(expected = ReadOnlyBufferException.class)
    public void buildByteBufReadOnly() throws Exception {
        int shardId = 111;

        TWriteRequest writeOne = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 222L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 1213))
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(writeOne);

        try (AccumulatedShardCommand command = accumulator.build()) {
            command.getReadOnlyBuffer().writeByte(1);
            fail("AccumulatedShardCommand should contain already ready only byteBuf, without it buffer can be damaged");
        }
    }

    @Test
    public void writeMultiple() throws Exception {
        int shardId = 333;
        accumulator = new ShardCommandAccumulator(shardId);

        TShardCommandRequest.Builder expect = TShardCommandRequest.newBuilder()
                .setShardId(shardId);

        for (int index = 0; index < 1000; index++) {
            TWriteRequest write = TWriteRequest.newBuilder()
                    .setMetricId(metricId(shardId, 1 + index))
                    .addPoints(TPoint.newBuilder()
                            .setTimestampsMillis(System.currentTimeMillis())
                            .setDoubleValue(Math.random())
                            .build())
                    .build();


            expect.addCommands(TCommandRequest.newBuilder().setWrite(write).build());
            accumulator.append(write);
        }

        TShardCommandRequest result = deserialize(accumulator.build());
        assertThat(result, equalTo(expect.build()));
    }

    @Test
    public void writeCompressed() throws Exception {
        int shardId = 555;

        byte[] content = new byte[1024 * 1024];
        ThreadLocalRandom.current().nextBytes(content);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 8585))
                .setType(MetricType.DGAUGE)
                .setDeadline(System.currentTimeMillis() + 1000)
                .setBinaryVersion(25)
                .addChunks(TimeSeries.Chunk.newBuilder()
                        .setFromMillis(timeToMillis("2017-06-21T15:54:26Z"))
                        .setToMillis(timeToMillis("2017-06-30T15:54:26Z"))
                        .setPointCount(10000)
                        .setContent(ByteString.copyFrom(content))
                        .build())
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(request);

        AccumulatedShardCommand commands = accumulator.build();
        TShardCommandRequest result = deserialize(commands);

        TShardCommandRequest expected = TShardCommandRequest.newBuilder()
                .setShardId(shardId)
                .addCommands(TCommandRequest.newBuilder().setCompressedWrite(request).build())
                .build();

        assertThat(result, equalTo(expected));
    }

    @Test
    public void resetWrites() throws Exception {
        int shardId = 123;

        TWriteRequest writeOne = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 1111L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:30:00Z", 3))
                .addPoints(point("2017-07-12T13:20:00Z", 2))
                .addPoints(point("2017-07-12T13:10:00Z", 1))
                .build();

        TWriteRequest writeTwo = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 2222L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T14:10:00Z", 1))
                .addPoints(point("2017-07-12T14:20:00Z", 2))
                .addPoints(point("2017-07-12T14:30:00Z", 3))
                .build();

        TWriteRequest writeTree = TWriteRequest.newBuilder()
                .setMetricId(metricId(shardId, 3333L))
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T15:30:00Z", 33))
                .build();

        accumulator = new ShardCommandAccumulator(shardId);
        accumulator.append(writeOne);

        int useBytesBeforeMark = accumulator.getCountBytes();
        int commandsCountBeforeMark = accumulator.getCountCommands();

        accumulator.markWriteIndex();
        accumulator.append(writeTwo);
        accumulator.append(writeTree);

        assertThat(accumulator.getCountBytes(), not(equalTo(useBytesBeforeMark)));
        assertThat(accumulator.getCountCommands(), not(equalTo(commandsCountBeforeMark)));

        accumulator.resetWriterIndex();

        assertThat(useBytesBeforeMark, equalTo(useBytesBeforeMark));
        assertThat(commandsCountBeforeMark, equalTo(commandsCountBeforeMark));

        accumulator.append(writeTree);

        TShardCommandRequest result = deserialize(accumulator.build());

        TShardCommandRequest expected = TShardCommandRequest.newBuilder()
                .setShardId(shardId)
                .addCommands(TCommandRequest.newBuilder().setWrite(writeOne).build())
                .addCommands(TCommandRequest.newBuilder().setWrite(writeTree).build())
                .build();

        assertThat(result, equalTo(expected));
    }

    private TShardCommandRequest deserialize(AccumulatedShardCommand commands) throws IOException {
        try {
            return TShardCommandRequest.parseFrom(new ByteBufInputStream(commands.getReadOnlyBuffer()));
        } finally {
            commands.close();
        }
    }
}
