package ru.yandex.stockpile.client.writeRequest;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Random;

import com.google.protobuf.CodedInputStream;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static junit.framework.TestCase.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.stockpile.api.EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS;
import static ru.yandex.stockpile.api.EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS;
import static ru.yandex.stockpile.api.EDecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS;
import static ru.yandex.stockpile.api.EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS;
import static ru.yandex.stockpile.api.EDecimPolicy.POLICY_KEEP_FOREVER;

/**
 * @author Sergey Polovko
 */
public class StockpileShardWriteRequestSerializerTest {

    private static final EDecimPolicy[] DECIM_POLICIES = {
        POLICY_KEEP_FOREVER,
        POLICY_5_MIN_AFTER_7_DAYS,
        POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS,
        POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS,
        POLICY_5_MIN_AFTER_2_MONTHS,
    };

    private final Random rnd = new Random();

    @Test
    public void serializeDeserialize() {
        try (var request = randomRequest(rnd, new Long2ObjectOpenHashMap<>())) {
            byte[] buf = StockpileShardWriteRequestSerializer.serialize(request);

            try (var deserialized = StockpileShardWriteRequestSerializer.deserialize(buf)) {
                Assert.assertEquals(request.getProjectId(), deserialized.getProjectId());
                Assert.assertEquals(request.getSerializerVersionId(), deserialized.getSerializerVersionId());
                Assert.assertEquals(request.getStockpileOwnerShardId(), deserialized.getStockpileOwnerShardId());
                Assert.assertEquals(request.getVersion(), deserialized.getVersion());
                Assert.assertEquals(request.getData(), deserialized.getData());
            }
        }
    }

    @Test
    public void serializeMultipleRequests() {
        var expected = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        byte[] buf;
        try (var r1 = randomRequest(rnd, expected);
             var r2 = randomRequest(rnd, expected);
             var r3 = randomRequest(rnd, expected))
        {
            buf = StockpileShardWriteRequestSerializer.serialize(List.of(r1, r2, r3));
        }

        var actual = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        StockpileShardWriteRequestSerializer.deserializeTo(CodedInputStream.newInstance(buf), actual);
        assertArchivesEqual(expected, actual);
    }

    @Test
    public void deserializeSingleRequest() {
        var expected = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        byte[] bytes = randomRequestSerialized(rnd, expected);

        var actual = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        StockpileShardWriteRequestSerializer.deserializeTo(CodedInputStream.newInstance(bytes), actual);
        assertArchivesEqual(expected, actual);
    }

    @Test
    public void deserializeMultipleRequest() throws IOException {
        var expected = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        ByteArrayOutputStream bytes = new ByteArrayOutputStream(1000);
        bytes.write(randomRequestSerialized(rnd, expected));
        bytes.write(randomRequestSerialized(rnd, expected));
        bytes.write(randomRequestSerialized(rnd, expected));

        var actual = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        StockpileShardWriteRequestSerializer.deserializeTo(CodedInputStream.newInstance(bytes.toByteArray()), actual);
        assertArchivesEqual(expected, actual);
    }

    private static byte[] randomRequestSerialized(Random rnd, Long2ObjectMap<MetricArchiveMutable> expected) {
        try (var request = randomRequest(rnd, expected)) {
            return request.serialize();
        }
    }

    private static StockpileShardWriteRequest randomRequest(Random rnd, Long2ObjectMap<MetricArchiveMutable> expected) {
        EProjectId projectId = OwnerField.random(rnd);
        int ownerShardId = rnd.nextInt();
        short decimPolicyId = (short) DECIM_POLICIES[rnd.nextInt(DECIM_POLICIES.length)].getNumber();
        var builder = new StockpileShardWriteRequestBuilder(projectId, ownerShardId);

        // 100 metrics with [10..20) points
        for (int i = 0; i < 100; i++) {
            long localId = StockpileLocalId.random();
            MetricType type = randomMetricType();

            var archive = new MetricArchiveMutable();
            archive.setOwnerProjectIdEnum(projectId);
            archive.setType(type);
            archive.setOwnerShardId(ownerShardId);
            archive.setDecimPolicyId(decimPolicyId);
            expected.put(localId, archive);

            for (int j = 0, count = 10 + rnd.nextInt(10); j < count; j++) {
                AggrPoint point = randomPoint(type);
                builder.addRecord(localId, point, decimPolicyId, type);
                archive.addRecord(point);
            }
        }

        // 100 metrics with completely delete
        for (int i = 0; i < 100; i++) {
            long localId = StockpileLocalId.random();
            builder.addDeleteData(localId);
            var archive = new MetricArchiveMutable();
            archive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
            expected.put(localId, archive);
        }

        // 100 metrics with delete before TS
        for (int i = 0; i < 100; i++) {
            long localId = StockpileLocalId.random();
            long deleteBeforeMillis = TsRandomData.randomTs(rnd);
            builder.addDeleteDataWithTs(localId, deleteBeforeMillis);
            var archive = new MetricArchiveMutable();
            archive.setDeleteBefore(deleteBeforeMillis);
            expected.put(localId, archive);
        }

        return builder.build();
    }

    private static void assertArchivesEqual(Long2ObjectMap<MetricArchiveMutable> expected, Long2ObjectMap<MetricArchiveMutable> actual) {
        assertEquals(expected.size(), actual.size());

        for (long localId : expected.keySet()) {
            MetricArchiveMutable expectedArchive = expected.get(localId);
            MetricArchiveMutable actualArchive = actual.get(localId);
            assertEquals(expectedArchive, actualArchive);
        }
    }
}
