package ru.yandex.stockpile.client.writeRequest;

import java.time.Instant;
import java.util.Random;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.stockpile.api.EProjectId;

import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.model.point.AggrPoints.point;

/**
 * @author Maksim Leonov (nohttp@)
 */
public class StockpileShardWriteRequestTest {

    @Test
    public void test() {
        Random r = new Random(20);

        for (int k = 0; k < 20; k++) {
            EProjectId projectId = r.nextBoolean() ? EProjectId.SOLOMON : EProjectId.GRAPHITE;
            var builder = new StockpileShardWriteRequestBuilder(projectId, 10);
            var reference = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
            for (int i = 0; i < 1000; i++) {
                long localId = r.nextLong();

                MetricType type = randomSupportedMetricType();
                for (int j = 0; j < r.nextInt(5); j++) {
                    MetricArchiveMutable referenceArchive = reference.get(localId);
                    if (referenceArchive == null) {
                        referenceArchive = new MetricArchiveMutable();
                        reference.put(localId, referenceArchive);
                    }
                    referenceArchive.setType(type);

                    AggrPoint point = randomPoint(randomMask(type));
                    short decimPolicyId = randomSupportedDecimPolicyId(r);
                    int deleteDataProbability = r.nextInt(100);

                    if (deleteDataProbability > 90) {
                        builder.addDeleteData(localId);
                        referenceArchive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
                    } else if (deleteDataProbability > 80) {
                        long tsMillis = point.getTsMillis();
                        builder.addDeleteDataWithTs(localId, tsMillis);
                        referenceArchive.setDeleteBefore(tsMillis);
                    } else {
                        builder.addRecord(localId, point, decimPolicyId, type);
                        referenceArchive.setDecimPolicyId(decimPolicyId);
                        referenceArchive.setOwnerProjectIdEnum(projectId);
                        referenceArchive.setOwnerShardId(10);
                        referenceArchive.addRecord(point);
                    }
                }
            }

            Long2ObjectOpenHashMap<MetricArchiveMutable> built;
            try (var request = builder.build()) {
                built = request.getDataByLocalId();
            }
            Assert.assertEquals(reference.size(), built.size());
            for (long localId : reference.keySet()) {
                Assert.assertEquals(reference.get(localId), built.get(localId));
            }
        }
    }

    @Test
    public void testDelete() {
        long ts = Instant.parse("2014-08-21T10:01:26.000Z").toEpochMilli();
        long localId = 100500L;

        var builder = new StockpileShardWriteRequestBuilder(EProjectId.TEST, 10);

        AggrPoint pointData = AggrPoint.shortPoint(ts, 1);
        AggrPoint pointNone = new AggrPoint();

        builder.addRecord(localId, pointData, 0, MetricType.DGAUGE);
        builder.addRecord(localId, pointNone, 0, MetricType.DGAUGE);
        builder.addRecord(localId, pointData, 0, MetricType.DGAUGE);

        Long2ObjectOpenHashMap<MetricArchiveMutable> content;
        try (var write = builder.build()) {
            content = write.getDataByLocalId();
        }
        AggrGraphDataArrayList graphData = content.get(localId).toAggrGraphDataArrayList();
        Assert.assertEquals(AggrGraphDataArrayList.of(point(ts, 1)), graphData);
    }

    private MetricType randomSupportedMetricType() {
        MetricType type;
        do {
            type = randomMetricType();
        } while (!CompressStreamFactory.isSupported(type));
        return type;
    }

    private short randomSupportedDecimPolicyId(Random r) {
        return (short) r.nextInt(DecimPoliciesPredefined.policiesCount());
    }
}
