package ru.yandex.stockpile.client.writeRequest.serializers;

import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import javax.annotation.Nonnull;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.hamcrest.CoreMatchers;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.lang.ShortUtils;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.serializer.HeapStockpileSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.writeRequest.serializers.flat.WriteRequestPointFlatDeserializer;
import ru.yandex.stockpile.client.writeRequest.serializers.flat.WriteRequestPointFlatSerializer;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.model.point.AggrPoints.dhistogram;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Maksim Leonov (nohttp@)
 */
public class WriteRequestPointSerializerDeserializerTest {
    private WriteRequestPointSerializer serializer;
    private WriteRequestPointDeserializer deserializer;

    @Before
    public void resetSerializerDeserializerState() {
        serializer = new WriteRequestPointFlatSerializer();
        deserializer = new WriteRequestPointFlatDeserializer();
    }

    @Test
    public void testFlatFormat() {
        testImpl();
    }

    @Test
    public void testFlatFormatDeleteData() {
        testDeleteDataImpl();
    }

    @Test
    public void serializeDeserializeDeleteData() {
        MetricArchiveMutable expect = new MetricArchiveMutable();
        expect.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        expect.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        expect.setOwnerShardId(10);

        var serializedBuffer = new HeapStockpileSerializer();
        serializedBuffer.writeBoolean(false);
        serializer.writeDeleteData(serializedBuffer, 123L);

        var content = deserialize(serializedBuffer.build());
        MetricArchiveMutable result = content.get(123L);

        Assert.assertThat(result.getDeleteBefore(), CoreMatchers.equalTo(expect.getDeleteBefore()));
        close(expect, result);
    }

    @Test
    public void ignoreSerializeEmptyPoints() {
        var serializedBuffer = new HeapStockpileSerializer();

        write(serializedBuffer, 123L, MetricType.METRIC_TYPE_UNSPECIFIED, new AggrPoint());

        var result = deserialize(serializedBuffer.build());
        MetricArchiveMutable archive = result.get(123L);

        Assert.assertTrue(archive.isEmpty());
        close(archive);
    }

    @Test
    public void serializeDeserializeOneHistogram() {
        var buffer = new HeapStockpileSerializer();

        AggrPoint point = point("2018-03-14T11:52:11.864Z",
                dhistogram(new double[]{5, 10, 30}, new long[]{3, 4, 0}));

        write(buffer, 123L, MetricType.HIST, point);

        var result = deserialize(buffer.build());
        AggrGraphDataArrayList list = result.get(123L).toAggrGraphDataArrayList();
        assertThat(list, equalTo(AggrGraphDataArrayList.of(point)));
    }

    @Test
    public void serializeDeserializeFewHistogram() {
        var buffer = new HeapStockpileSerializer();

        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2018-03-14T11:52:00Z", dhistogram(new double[]{5, 10, 30}, new long[]{3, 4, 0})),
                point("2018-03-14T11:53:00Z", dhistogram(new double[]{5, 10, 30}, new long[]{3, 4, 0})),
                point("2018-03-14T11:54:00Z", dhistogram(new double[]{10, 20, 30, 40}, new long[]{4, 3, 2, 1}))
        );

        for (int index = 0; index < source.length(); index++) {
            AggrPoint point = source.getAnyPoint(index);
            write(buffer, 123L, MetricType.HIST, point);
        }

        var result = deserialize(buffer.build());
        AggrGraphDataArrayList list = result.get(123L).toAggrGraphDataArrayList();
        assertThat(list, equalTo(source));
    }

    private Long2ObjectOpenHashMap<MetricArchiveMutable> deserialize(byte[] data) {
        StockpileDeserializer deserializeBuffer = new StockpileDeserializer(data);
        var result = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
        while (!deserializeBuffer.atEof()) {
            if (deserializeBuffer.readBoolean()) {
                deserializer.readValueTo(result, deserializeBuffer, EProjectId.SOLOMON, 10);
            } else {
                deserializer.readDeleteDataTo(result, deserializeBuffer);
            }
        }

        return result;
    }

    private void testImpl() {
        Random r = new Random(20);

        for (int i = 0; i < 1000; i++) {
            // clean up previous iteration
            resetSerializerDeserializerState();

            EProjectId projectId = randomProjectId(r);

            var stockpileSerializer = new HeapStockpileSerializer();
            var generated = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
            var reference =  new Long2ObjectOpenHashMap<MetricArchiveMutable>();

            Map<Long, MetricType> localIdToType = new HashMap<>();
            for (int k = 0; k < 200; k++) {
                long localId = randomLocalId(r);
                short decimPolicyId = randomPolicy(r);
                boolean deleteData = randomDeleteData(r);

                MetricType type = localIdToType.computeIfAbsent(localId, ignore -> {
                    MetricType result;
                    do {
                        result = randomMetricType();
                    } while (!CompressStreamFactory.isSupported(result));
                    return result;
                });
                AggrPoint point = randomPoint(type);

                if (deleteData) {
                    stockpileSerializer.writeBoolean(false);
                    serializer.writeDeleteData(stockpileSerializer, localId);
                } else {
                    stockpileSerializer.writeBoolean(true);
                    serializer.writeValue(
                        stockpileSerializer,
                        localId,
                        decimPolicyId,
                        type,
                        point.columnSetMask(),
                        point
                    );
                }

                MetricArchiveMutable referenceArchive = reference.computeIfAbsent(localId, ignore -> new MetricArchiveMutable());
                if (deleteData) {
                    referenceArchive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
                } else {
                    referenceArchive.setDecimPolicyId(decimPolicyId);
                    referenceArchive.setOwnerProjectIdEnum(projectId);
                    referenceArchive.setOwnerShardId(10);
                    referenceArchive.setType(type);

                    referenceArchive.addRecord(point);
                }
            }
            StockpileDeserializer stockpileDeserializer = new StockpileDeserializer(stockpileSerializer.buildByteString());
            while (!stockpileDeserializer.atEof()) {
                if (stockpileDeserializer.readBoolean()) {
                    deserializer.readValueTo(generated, stockpileDeserializer, projectId, 10);
                } else {
                    deserializer.readDeleteDataTo(generated, stockpileDeserializer);
                }
            }
            Assert.assertThat(generated, CoreMatchers.equalTo(reference));
        }
    }

    private void testDeleteDataImpl() {
        Random r = new Random(20);

        for (int i = 0; i < 1000; i++) {
            resetSerializerDeserializerState();

            var stockpileSerializer = new HeapStockpileSerializer();
            var generated = new Long2ObjectOpenHashMap<MetricArchiveMutable>();
            var reference = new Long2ObjectOpenHashMap<MetricArchiveMutable>();

            for (int k = 0; k < 200; k++) {
                long localId = randomLocalId(r);

                stockpileSerializer.writeBoolean(false);
                serializer.writeDeleteData(
                    stockpileSerializer,
                    localId
                );
                var archive = new MetricArchiveMutable();
                archive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
                reference.put(localId, archive);
            }
            StockpileDeserializer stockpileDeserializer = new StockpileDeserializer(stockpileSerializer.buildByteString());
            while (!stockpileDeserializer.atEof()) {
                Assert.assertFalse(stockpileDeserializer.readBoolean());
                deserializer.readDeleteDataTo(generated, stockpileDeserializer);
            }

            Assert.assertEquals(reference, generated);
        }
    }

    @Nonnull
    private static EProjectId randomProjectId(Random r) {
        return r.nextBoolean() ? EProjectId.SOLOMON : EProjectId.GRAPHITE;
    }

    private static long randomLocalId(Random r) {
        if (r.nextBoolean()) {
            return StockpileLocalId.MIN_VALID + r.nextInt(10);
        } else {
            for (;;) {
                long localId = r.nextLong();
                if (StockpileLocalId.isValid(localId)) {
                    return localId;
                }
            }
        }
    }

    private static short randomPolicy(Random r) {
        if (r.nextBoolean()) {
            return ShortUtils.toShortExact(EDecimPolicy.POLICY_KEEP_FOREVER.getNumber());
        } else {
            return ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        }
    }

    private static boolean randomDeleteData(Random r) {
        return r.nextInt(100) < 5;
    }

    private void write(StockpileSerializer s, long localId, MetricType type, AggrPoint point) {
        s.writeBoolean(true); // indicate one additional point
        serializer.writeValue(s, localId, 0, type, point.columnSet, point);
    }
}
