package ru.yandex.solomon.tracing;

import java.util.Arrays;
import java.util.stream.Collectors;

import io.jaegertracing.Configuration.CodecConfiguration;
import io.jaegertracing.Configuration.Propagation;
import io.jaegertracing.Configuration.ReporterConfiguration;
import io.jaegertracing.Configuration.SamplerConfiguration;
import io.jaegertracing.Configuration.SenderConfiguration;
import io.jaegertracing.internal.JaegerTracer;
import io.jaegertracing.internal.samplers.RemoteControlledSampler;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Sergey Polovko
 */
@Configuration
public class TracingContext {

    @Bean
    TracingConfig tracingConfig(@Qualifier("ClientId") String clientId) {
        return new TracingConfig(clientId, Labels.empty());
    }

    @Bean
    Tracer tracer(TracingConfig config, MetricRegistry registry) {
        var builder = new io.jaegertracing.Configuration(config.getServiceName())
                .withMetricsFactory(new MonlibMetricsFactory(registry))
                .withSampler(SamplerConfiguration.fromEnv()
                        .withType(RemoteControlledSampler.TYPE)
                        .withParam(0.001))  // sample ~ 0.1% of all traces by default
                .withReporter(ReporterConfiguration.fromEnv()
                        .withLogSpans(Boolean.FALSE)
                        .withSender(SenderConfiguration.fromEnv()))
                .withTracerTags(config.getLabels().toMap())
                .withCodec(CodecConfiguration.fromString(Arrays.stream(Propagation.values())
                        .map(Enum::name)
                        .collect(Collectors.joining(","))));

        JaegerTracer tracer = builder.getTracer();
        GlobalTracer.registerIfAbsent(tracer);
        return tracer;
    }
}
