package ru.yandex.solomon.codec;

import java.util.DoubleSummaryStatistics;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.IntToDoubleFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class AggregationBenchmark {

    private static void test(String name, List<Statistic> summaries) {
        final int doubleMask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask() | StockpileColumn.MERGE.mask() | StockpileColumn.COUNT.mask();
        final int summaryMask = StockpileColumn.TS.mask() | StockpileColumn.DSUMMARY.mask() | StockpileColumn.MERGE.mask() | StockpileColumn.COUNT.mask();
        var d = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, doubleMask);
        var s = CompressStreamFactory.createOutputStream(MetricType.DSUMMARY, summaryMask);

        AggrPoint point = new AggrPoint();
        point.merge = true;
        point.count = 100;
        point.tsMillis = System.currentTimeMillis();
        for (var summary : summaries) {
            point.valueNum = summary.getSum();
            point.count = summary.getCount();
            point.summaryDouble = summary;
            d.writePoint(doubleMask, point);
            s.writePoint(summaryMask, point);
        }

        double b = 1.0 * d.getCompressedData().readableBits() / summaries.size();
        double b2 = 1.0 * s.getCompressedData().readableBits() / summaries.size();
        System.out.printf("%-10s: %.1f %.1f bits per point\n", name, b, b2);
    }


    private static long junk() {
        return ThreadLocalRandom.current().nextLong();
    }

    private static List<Statistic> gen(IntToDoubleFunction supplier) {
        return IntStream.range(0, 500)
            .mapToObj(pointIndex -> IntStream.range(pointIndex, pointIndex + 100)
                .mapToDouble(supplier)
                .collect(Statistic::new, Statistic::accept, Statistic::combine))
            .collect(Collectors.toList());
    }

    private static void test(String name, IntToDoubleFunction supplier) {
        var source = gen(supplier);
        test(name, source);
    }

    public static void main(String[] args) {
        Random rnd = ThreadUtils.currentThreadLocalRandom();
        test("0s", i -> 0);
        test("1s", i -> 1);
        test("rnd 0,1", i -> rnd.nextBoolean() ? 1 : 0);
        test("rnd 1,2", i -> rnd.nextBoolean() ? 1 : 2);
        test("rnd 2,3", i -> rnd.nextBoolean() ? 2 : 3);
        test("0,1", i -> i % 2);
        test("1,2", i -> 1 + i % 2);
        test("0..1000", i -> rnd.nextInt(1000));
        test("1500..1600", i -> rnd.nextInt(100) + 1500);
        test("0..1000000", i -> rnd.nextInt(1000000));
        test("junk", i -> junk());
        test("grow", i -> i);
    }

    private static class Statistic extends DoubleSummaryStatistics implements SummaryDoubleSnapshot {
        double last;

        @Override
        public void accept(double value) {
            this.last = value;
            super.accept(value);
        }

        void combine(Statistic other) {
            last = other.last;
            super.combine(other);
        }

        @Override
        public double getLast() {
            return last;
        }
    }

    /*

0s        : 4.4 6.4 bits per point
1s        : 4.4 8.6 bits per point
rnd 0,1   : 10.1 20.0 bits per point
rnd 1,2   : 22.2 32.6 bits per point
rnd 2,3   : 14.3 24.9 bits per point
0,1       : 4.4 19.5 bits per point
1,2       : 4.4 20.6 bits per point
0..1000   : 31.4 72.8 bits per point
1500..1600: 33.4 64.4 bits per point
0..1000000: 41.4 113.9 bits per point
junk      : 73.5 261.3 bits per point
grow      : 21.0 50.5 bits per point

     */


}
