package ru.yandex.solomon.codec;

import java.util.concurrent.ThreadLocalRandom;

import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
public class CompressSizeBenchmark {
    private static void test(MetricType type, String name) {
        PointValueDataSets.PointValueGenerator supplier = PointValueDataSets.gen(type, name);
        final int mask = StockpileColumns.minColumnSet(type);
        TimeSeriesOutputStream stream =
                CompressStreamFactory.createOutputStream(type, mask);

        AggrPoint point = new AggrPoint(mask);
        point.tsMillis = System.currentTimeMillis();
        ThreadLocalRandom random = ThreadLocalRandom.current();
        for (int index = 0; index < 500; index++) {
            supplier.fill(index, point, random);
            stream.writePoint(mask, point);
        }

        long lengthBits = stream.getCompressedData().readableBits();
        double b = 1.0 * lengthBits / 500;
        double b2 = 1.0 * (lengthBits - 64) / (500 - 1);
        System.out.printf("%-15s: %-5.1f %-5.1f bits per point\n", name, b, b2);
    }

    public static void main(String[] args) {
        for (MetricType type : MetricType.values()) {
            if (!CompressStreamFactory.isSupported(type)) {
                continue;
            }

            System.out.println(type);
            System.out.println("====");
            try {
                for (String name : PointValueDataSets.GENERATOR_BY_NAME.keySet()) {
                    test(type, name);
                }
            } catch (UnsupportedOperationException e) {
                System.out.println(e.getMessage());
            }
            System.out.println();
        }
    }
}

/*
DGAUGE
====
default        : 3.3   3.1   bits per point
1s             : 3.3   3.2   bits per point
rnd 0,1        : 8.8   8.7   bits per point
rnd 1,2        : 9.2   9.0   bits per point
rnd 2,3        : 4.3   4.2   bits per point
0,1            : 14.3  14.2  bits per point
1,2            : 15.3  15.2  bits per point
0..1000        : 24.3  24.2  bits per point
1500..1600     : 14.2  14.1  bits per point
1000000        : 28.2  28.2  bits per point
seq            : 11.4  11.2  bits per point
seqMultiply    : 21.1  21.0  bits per point
rnd            : 59.6  59.6  bits per point
rndGrow        : 71.2  71.2  bits per point

IGAUGE
====
default        : 3.3   3.1   bits per point
1s             : 3.3   3.1   bits per point
rnd 0,1        : 5.7   5.6   bits per point
rnd 1,2        : 5.7   5.6   bits per point
rnd 2,3        : 6.0   5.8   bits per point
0,1            : 8.2   8.1   bits per point
1,2            : 8.3   8.1   bits per point
0..1000        : 16.9  16.8  bits per point
1500..1600     : 12.5  12.4  bits per point
1000000        : 31.6  31.6  bits per point
seq            : 8.3   8.1   bits per point
seqMultiply    : 17.5  17.4  bits per point
rnd            : 59.2  59.2  bits per point
rndGrow        : 74.3  74.3  bits per point

COUNTER
====
default        : 3.3   3.1   bits per point
1s             : 3.3   3.2   bits per point
rnd 0,1        : 7.0   6.9   bits per point
rnd 1,2        : 7.0   6.9   bits per point
rnd 2,3        : 6.6   6.5   bits per point
0,1            : 8.2   8.1   bits per point
1,2            : 8.2   8.1   bits per point
0..1000        : 17.4  17.3  bits per point
1500..1600     : 13.1  13.0  bits per point
1000000        : 31.9  31.8  bits per point
seq            : 3.3   3.1   bits per point
seqMultiply    : 8.2   8.1   bits per point
rnd            : 65.9  65.9  bits per point
rndGrow        : 74.3  74.3  bits per point

RATE
====
default        : 3.3   3.1   bits per point
1s             : 3.3   3.2   bits per point
rnd 0,1        : 7.1   7.0   bits per point
rnd 1,2        : 7.0   6.9   bits per point
rnd 2,3        : 7.1   7.0   bits per point
0,1            : 8.2   8.1   bits per point
1,2            : 8.2   8.1   bits per point
0..1000        : 17.6  17.5  bits per point
1500..1600     : 13.1  13.0  bits per point
1000000        : 31.9  31.8  bits per point
seq            : 3.3   3.1   bits per point
seqMultiply    : 8.2   8.1   bits per point
rnd            : 65.3  65.3  bits per point
rndGrow        : 74.3  74.3  bits per point
 */
