package ru.yandex.solomon.codec;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListViewIterator;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 2, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class CompressSpeedJmhBenchmark {
    @Param({"1000"})
    public int pointCount;

    @Param({"DGAUGE", "COUNTER", "IGAUGE"})
    public MetricType type;

    @Param({"rnd", "seq", "rndGrow"})
    private String set;

    private AggrGraphDataArrayList source;
    private BitBuf compressed;

    /*

Benchmark                              (type)  (pointCount)    (set)   Mode  Cnt      Score     Error  Units
CompressSpeedJmhBenchmark.compress     DGAUGE          1000      rnd  thrpt   10  12736.100 ± 360.105  ops/s
CompressSpeedJmhBenchmark.compress     DGAUGE          1000      seq  thrpt   10  16066.040 ± 246.642  ops/s
CompressSpeedJmhBenchmark.compress     DGAUGE          1000  rndGrow  thrpt   10  11358.984 ± 230.502  ops/s
CompressSpeedJmhBenchmark.compress    COUNTER          1000      rnd  thrpt   10  13661.861 ± 386.988  ops/s
CompressSpeedJmhBenchmark.compress    COUNTER          1000      seq  thrpt   10  20966.682 ± 473.265  ops/s
CompressSpeedJmhBenchmark.compress    COUNTER          1000  rndGrow  thrpt   10  14181.939 ± 266.222  ops/s
CompressSpeedJmhBenchmark.compress     IGAUGE          1000      rnd  thrpt   10  13568.581 ± 229.996  ops/s
CompressSpeedJmhBenchmark.compress     IGAUGE          1000      seq  thrpt   10  16039.274 ± 224.826  ops/s
CompressSpeedJmhBenchmark.compress     IGAUGE          1000  rndGrow  thrpt   10  14934.959 ± 166.032  ops/s
CompressSpeedJmhBenchmark.decompress   DGAUGE          1000      rnd  thrpt   10  13911.517 ± 485.129  ops/s
CompressSpeedJmhBenchmark.decompress   DGAUGE          1000      seq  thrpt   10  19958.694 ± 653.894  ops/s
CompressSpeedJmhBenchmark.decompress   DGAUGE          1000  rndGrow  thrpt   10  12463.370 ± 302.483  ops/s
CompressSpeedJmhBenchmark.decompress  COUNTER          1000      rnd  thrpt   10  12000.450 ± 362.578  ops/s
CompressSpeedJmhBenchmark.decompress  COUNTER          1000      seq  thrpt   10  23776.049 ± 718.017  ops/s
CompressSpeedJmhBenchmark.decompress  COUNTER          1000  rndGrow  thrpt   10  11672.587 ± 187.799  ops/s
CompressSpeedJmhBenchmark.decompress   IGAUGE          1000      rnd  thrpt   10  12044.261 ± 173.941  ops/s
CompressSpeedJmhBenchmark.decompress   IGAUGE          1000      seq  thrpt   10  20398.644 ± 591.259  ops/s
CompressSpeedJmhBenchmark.decompress   IGAUGE          1000  rndGrow  thrpt   10  11699.018 ± 385.926  ops/s

     */


    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(CompressSpeedJmhBenchmark.class.getName())
                .detectJvmArgs()
                .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        int mask = StockpileColumns.minColumnSet(type);
        source = new AggrGraphDataArrayList(mask, pointCount);
        long stepMillis = TimeUnit.SECONDS.toMillis(15L);
        AggrPoint point = new AggrPoint();
        PointValueDataSets.PointValueGenerator generator = PointValueDataSets.gen(type, set);
        ThreadLocalRandom random = ThreadLocalRandom.current();
        long now = System.currentTimeMillis();
        for (int index = 0; index < pointCount; index++) {
            now += stepMillis;
            point.tsMillis = now;
            generator.fill(index, point, random);
            source.addRecordData(mask, point);
        }

        compressed = compress(source);
    }

    private BitBuf compress(AggrGraphDataArrayList source) {
        TimeSeriesOutputStream stream =
                CompressStreamFactory.createOutputStream(type,
                        source.columnSetMask(),
                        source.length());

        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayListViewIterator it = source.iterator();
        while (it.next(point)) {
            stream.writePoint(source.columnSetMask(), point);
        }
        return stream.getCompressedData();
    }

    private AggrGraphDataArrayList decompress(int mask, BitBuf view) {
        TimeSeriesInputStream stream =
                CompressStreamFactory.createInputStream(
                        type,
                        mask,
                        view);

        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(mask, pointCount);
        while (stream.hasNext()) {
            stream.readPoint(mask, point);
            result.addRecordData(mask, point);
        }
        return result;
    }

    @Benchmark
    public Object compress() {
        return compress(source);
    }

    @Benchmark
    public Object decompress() {
        return decompress(source.columnSetMask(), compressed);
    }
}
