package ru.yandex.solomon.codec;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.bits.NettyBitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class DecoderJmh {
    private static int POINTS = 100000;
    private static int MASK = TsColumn.mask | ValueColumn.mask;

    @Param({"NettyHeap", "NettyDirect", "NettyMax", "Array"})
    private String impl;
    private AggrGraphDataArrayList source;
    private BitBuf buffer;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(DecoderJmh.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx3g", "-Xms3g")
//            .addProfiler(FlightRecorderProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        if (source == null) {
            source = new AggrGraphDataArrayList(MASK, POINTS);

            long now = System.currentTimeMillis();
            var random = ThreadLocalRandom.current();
            AggrPoint point = new AggrPoint();
            point.tsMillis = now;
            point.valueNum = 1;
            for (int index = 0; index < POINTS; index++) {
                point.tsMillis += 15_000;
                point.valueNum = ValueRandomData.randomNum(random);
                source.addRecord(point);
            }
        }

        if (buffer != null) {
            buffer.release();
        }

        int capacity = 10 << 20; // 10 MiB
        switch (impl) {
            case "NettyHeap": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.heapBuffer(capacity), 0);
                break;
            }
            case "NettyDirect": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.directBuffer(capacity), 0);
                break;
            }
            case "NettyMax": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.heapBuffer(capacity, capacity), 0);
                break;
            }
            default:
                buffer = new HeapBitBuf(new byte[capacity], 0);
        }

        try (var out = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, MASK, buffer.retain(), 0)) {
            var point = RecyclableAggrPoint.newInstance();
            for (int index = 0; index < POINTS; index++) {
                source.getDataTo(index, point);
                out.writePoint(MASK, point);
            }
            point.recycle();
        }
    }

    @Benchmark
    public Object decode() {
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(MASK, POINTS);
        try (var in = CompressStreamFactory.createInputStream(MetricType.DGAUGE, MASK, buffer.duplicate())) {
            var point = RecyclableAggrPoint.newInstance();
            while (in.hasNext()) {
                in.readPoint(MASK, point);
                result.addRecord(point);
            }
            point.recycle();
        }
        return result;
    }
}
