package ru.yandex.solomon.codec;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.bits.NettyBitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class EncodeJmh {
    private static int POINTS = 100000;
    private static int MASK = TsColumn.mask | ValueColumn.mask;

//    @Param({"NettyHeap", "NettyDirect", "NettyMax", "Array"})
    @Param({"NettyHeap", "Array"})
    private String impl;
    private AggrGraphDataArrayList source;
    private BitBuf buffer;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(EncodeJmh.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx3g", "-Xms3g", "-Dio.netty.buffer.checkAccessible=false")
//            .addProfiler(FlightRecorderProfiler.class)
//            .addProfiler(AsyncProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        if (source == null) {
            source = new AggrGraphDataArrayList(MASK, POINTS);

            long now = System.currentTimeMillis();
            var random = ThreadLocalRandom.current();
            AggrPoint point = new AggrPoint();
            point.tsMillis = now;
            point.valueNum = 1;
            for (int index = 0; index < POINTS; index++) {
                point.tsMillis += 15_000;
                point.valueNum = ValueRandomData.randomNum(random);
                source.addRecord(point);
            }
        }

        if (buffer != null) {
            buffer.release();
        }

        int capacity = 10 << 20; // 10 MiB
        switch (impl) {
            case "NettyHeap": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.heapBuffer(capacity), 0);
                break;
            }
            case "NettyDirect": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.directBuffer(capacity), 0);
                break;
            }
            case "NettyMax": {
                buffer = new NettyBitBuf(ByteBufAllocator.DEFAULT.heapBuffer(capacity, capacity), 0);
                break;
            }
            default:
                buffer = new HeapBitBuf(new byte[capacity], 0);
        }
    }

    @Setup(Level.Invocation)
    public void before() {
        buffer.resetReadIndex();
        buffer.resetWriterIndex();
    }

    @Benchmark
    public Object encode() {
        var point = RecyclableAggrPoint.newInstance();
        try (var out = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, MASK, buffer.retain(), 0)) {
            for (int index = 0; index < POINTS; index++) {
                source.getDataTo(index, point);
                out.writePoint(MASK, point);
            }
            point.recycle();
            return out.getCompressedData();
        }
    }
}
