package ru.yandex.solomon.codec;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 5, time = 1, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 1, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class FramedTimeSeriesBenchmark {
    private static StockpileFormat FORMAT = StockpileFormat.CURRENT;
    private static int MASK = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();

    @Param({"1", "10", "100", "200", "1000"})
    private long pointAtFrame = 0;
    private BitBuf compressed;

    /*

Benchmark                                    (pointAtFrame)  Mode  Cnt      Score      Error  Units
FramedTimeSeriesBenchmark.iterateForRestore               1  avgt    5    188.639 ±    9.799  ns/op
FramedTimeSeriesBenchmark.iterateForRestore              10  avgt    5    816.365 ±  129.854  ns/op
FramedTimeSeriesBenchmark.iterateForRestore             100  avgt    5   5797.141 ± 1178.136  ns/op
FramedTimeSeriesBenchmark.iterateForRestore             200  avgt    5  10886.009 ±  908.422  ns/op
FramedTimeSeriesBenchmark.iterateForRestore            1000  avgt    5  57538.122 ± 6991.436  ns/op
FramedTimeSeriesBenchmark.restore                         1  avgt    5    222.294 ±   99.877  ns/op
FramedTimeSeriesBenchmark.restore                        10  avgt    5    263.350 ±   88.485  ns/op
FramedTimeSeriesBenchmark.restore                       100  avgt    5    339.987 ±   54.167  ns/op
FramedTimeSeriesBenchmark.restore                       200  avgt    5    344.907 ±  106.619  ns/op
FramedTimeSeriesBenchmark.restore                      1000  avgt    5   1096.159 ±  349.018  ns/op

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(FramedTimeSeriesBenchmark.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx3g", "-Xms3g")
//            .addProfiler(FlightRecorderProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        var out = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, MASK);

        long now = System.currentTimeMillis();
        AggrPoint point = new AggrPoint();
        point.tsMillis = now;
        point.valueNum = 1;
        for (int index = 0; index < pointAtFrame; index++) {
            point.tsMillis += 10_000;
            point.valueNum = ThreadLocalRandom.current().nextInt(1, 100);
            out.writePoint(MASK, point);
        }
        out.forceCloseFrame();
        compressed = out.getCompressedData();
    }

    @Benchmark
    public TimeSeriesOutputStream restore() {
        return CompressStreamFactory.restoreOutputStream(MetricType.DGAUGE, MASK, compressed.asReadOnly());
    }

    @Benchmark
    public TimeSeriesOutputStream iterateForRestore() {
        var out = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, MASK);
        out.ensureCapacity(compressed.bytesSize());
        var it = CompressStreamFactory.createInputStream(MetricType.DGAUGE, MASK, compressed.asReadOnly());
        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        while (it.hasNext()) {
            it.readPoint(MASK, point);
            out.writePoint(MASK, point);
        }
        point.recycle();
        return out;
    }
}
