package ru.yandex.solomon.codec;

import java.util.concurrent.TimeUnit;

import io.netty.util.internal.ThreadLocalRandom;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.TimeFilterAggrGraphDataIterator;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 1, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 1, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class MetricArchiveCropJmh {
    @Param({"1", "10", "100"})
    private int hours;

    protected ThreadLocalRandom random = ThreadUtils.currentThreadLocalRandom();
    protected AggrPoint point;
    private MetricArchiveMutable archive;
    private long fromTsMillis;
    private long toTsMillis;

    /*

Benchmark                               (hours)  Mode  Cnt    Score    Error  Units
MetricArchiveCropJmh.toListIterators          1  avgt   10  198.398 ± 12.436  ns/op
MetricArchiveCropJmh.toListIterators         10  avgt   10  190.832 ± 28.393  ns/op
MetricArchiveCropJmh.toListIterators        100  avgt   10  184.760 ± 17.677  ns/op
MetricArchiveCropJmh.toListViewAndCrop        1  avgt   10  359.690 ± 26.097  ns/op
MetricArchiveCropJmh.toListViewAndCrop       10  avgt   10  356.081 ± 31.498  ns/op
MetricArchiveCropJmh.toListViewAndCrop      100  avgt   10  344.881 ± 22.882  ns/op

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(MetricArchiveCropJmh.class.getName())
//                .addProfiler(GCProfiler.class)
//                .addProfiler(FlightRecorderProfiler.class)
//                .addProfiler(LinuxPerfProfiler.class)
            .detectJvmArgs()
            .jvmArgs("-Xmx3g", "-Xms3g")
            .build();

        new Runner(opt).run();
    }

    @Setup(Level.Trial)
    public void iterationSetUp() {
        archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        long now = System.currentTimeMillis();

        ThreadLocalRandom random = ThreadLocalRandom.current();
        AggrPoint point = new AggrPoint();
        point.setTsMillis(now - hours);
        point.setValue(42);
        archive.addRecord(point);
        while (archive.getLastTsMillis() < now) {
            randomPoint(point, archive.columnSetMask(), random);
            point.tsMillis = archive.getLastTsMillis() + 15_000;
            archive.addRecord(point);
        }

        fromTsMillis = now - TimeUnit.HOURS.toMillis(1);
        toTsMillis = now;
    }

    @Benchmark
    public Object toListViewAndCrop() {
        return archive.toAggrGraphDataArrayList()
            .view()
            .cropForResponse(fromTsMillis, toTsMillis, false);
    }

    @Benchmark
    public Object toListIterators() {
        return AggrGraphDataArrayList.of(TimeFilterAggrGraphDataIterator.slice(archive.iterator(), fromTsMillis, toTsMillis));
    }
}
