package ru.yandex.solomon.codec;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import io.netty.util.internal.ThreadLocalRandom;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 15, time = 3, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 3, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public abstract class MetricArchiveMutableAddBenchmark {
    protected ThreadLocalRandom random = ThreadUtils.currentThreadLocalRandom();
    protected AggrPoint point;
    private MetricArchiveMutable archive;

    public static void main(String[] args) throws RunnerException, IOException {
        Options opt = new OptionsBuilder()
                .include(MetricArchiveMutableAddBenchmark.class.getName())
//                .exclude(".*LogarithmHistogram")
//                .exclude(".*Double")
//                .addProfiler(GCProfiler.class)
//                .addProfiler(FlightRecorderProfiler.class)
//                .addProfiler(LinuxPerfProfiler.class)
                .detectJvmArgs()
                .jvmArgs("-Xmx3g", "-Xms3g")
                .build();

        new Runner(opt).run();
    }

    @Setup(Level.Iteration)
    public void iterationSetUp() throws IOException {
        archive = new MetricArchiveMutable();
        point = new AggrPoint();
    }

    @Benchmark
    public AggrPoint baseline() {
        return nextPoint();
    }

    @Benchmark
    public int addPointToArchiveContentOld() {
        archive.addRecord(nextPoint());
        return archive.getRecordCount();
    }

    abstract AggrPoint nextPoint();

    public static class Double extends MetricArchiveMutableAddBenchmark {
        private long timeMillis = System.currentTimeMillis();

        @Override
        AggrPoint nextPoint() {
            timeMillis += random.nextInt(15000);
            point.setTsMillis(timeMillis);
            point.setValue(ValueRandomData.randomNum(random), ValueColumn.DEFAULT_DENOM);
            return point;
        }
    }


    public static class LogarithmHistogram extends MetricArchiveMutableAddBenchmark {
        private long timeMillis = System.currentTimeMillis();

        @Override
        AggrPoint nextPoint() {
            LogHistogram previous = point.logHistogram;
            if (previous == null) {
                previous = LogHistogram.newInstance();
            }

            var builder = LogHistogram.newBuilder().setMaxBucketsSize(10);

            if (random.nextBoolean()) {
                builder.setCountZero(previous.getCountZero());
            } else {
                builder.setCountZero(random.nextLong(0, 10000));
            }

            if (random.nextBoolean()) {
                builder.setStartPower(previous.getStartPower());
            } else {
                builder.setStartPower(random.nextInt(-100, 100));
            }

            if (random.nextBoolean()) {
                builder.setBuckets(previous.copyBuckets());
            } else {
                double[] buckets = new double[10];
                for (int index = 0; index < buckets.length; index++) {
                    buckets[index] = ValueRandomData.randomNum(random);
                }
                builder.setBuckets(buckets);
            }

            timeMillis += random.nextInt(15000);
            point.setTsMillis(timeMillis);
            point.setLogHistogram(builder.build());
            return point;
        }
    }

        /*

Benchmark                                                                                          Mode  Cnt     Score     Error   Units
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent                                   avgt   30   222.164 ±   7.576   ns/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·cpi                              avgt          0.395               CPI
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.alloc.rate                    avgt   30  1405.109 ±  42.212  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.alloc.rate.norm               avgt   30   326.818 ±   1.332    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Eden_Space           avgt   30  1384.803 ± 119.771  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Eden_Space.norm      avgt   30   322.144 ±  25.862    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Old_Gen              avgt   30   110.114 ±  72.164  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Old_Gen.norm         avgt   30    25.841 ±  16.954    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Survivor_Space       avgt   30     0.027 ±   0.011  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.churn.PS_Survivor_Space.norm  avgt   30     0.006 ±   0.003    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.count                         avgt   30   114.000            counts
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveContent:·gc.time                          avgt   30  1703.000                ms
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream                                    avgt   30   137.833 ±   1.485   ns/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·cpi                               avgt          0.396               CPI
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.alloc.rate                     avgt   30  1467.902 ±  13.289  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.alloc.rate.norm                avgt   30   212.133 ±   0.339    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Eden_Space            avgt   30  1463.994 ± 156.925  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Eden_Space.norm       avgt   30   211.848 ±  23.803    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Old_Gen               avgt   30   214.064 ±  97.501  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Old_Gen.norm          avgt   30    31.033 ±  14.149    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Survivor_Space        avgt   30     0.033 ±   0.007  MB/sec
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.churn.PS_Survivor_Space.norm   avgt   30     0.005 ±   0.001    B/op
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.count                          avgt   30   128.000            counts
MetricArchiveMutableAddBenchmark.Double.addPointToArchiveStream:·gc.time                           avgt   30  2920.000                ms
MetricArchiveMutableAddBenchmark.Double.baseline                                                   avgt   30    58.391 ±   0.390   ns/op
MetricArchiveMutableAddBenchmark.Double.baseline:·cpi                                              avgt          0.435               CPI
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.alloc.rate                                    avgt   30  2874.504 ±  19.180  MB/sec
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.alloc.rate.norm                               avgt   30   176.000 ±   0.001    B/op
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.churn.PS_Eden_Space                           avgt   30  2877.645 ±  49.265  MB/sec
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.churn.PS_Eden_Space.norm                      avgt   30   176.192 ±   2.751    B/op
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.churn.PS_Survivor_Space                       avgt   30     0.093 ±   0.018  MB/sec
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.churn.PS_Survivor_Space.norm                  avgt   30     0.006 ±   0.001    B/op
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.count                                         avgt   30   622.000            counts
MetricArchiveMutableAddBenchmark.Double.baseline:·gc.time                                          avgt   30   745.000                ms


Benchmark                                                                                                      Mode  Cnt     Score     Error   Units
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent                                   avgt   30   796.288 ±   5.830   ns/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·cpi                              avgt          0.425               CPI
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.alloc.rate                    avgt   30  1531.841 ±  10.404  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.alloc.rate.norm               avgt   30  1279.479 ±   0.932    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Eden_Space           avgt   30  1472.233 ± 148.643  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Eden_Space.norm      avgt   30  1230.199 ± 126.341    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Old_Gen              avgt   30   103.396 ±  72.899  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Old_Gen.norm         avgt   30    86.568 ±  61.032    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Survivor_Space       avgt   30     0.023 ±   0.012  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.churn.PS_Survivor_Space.norm  avgt   30     0.020 ±   0.010    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.count                         avgt   30   123.000            counts
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveContent:·gc.time                          avgt   30  1797.000                ms
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream                                    avgt   30   737.542 ±   3.664   ns/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·cpi                               avgt          0.429               CPI
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.alloc.rate                     avgt   30  1537.979 ±  33.101  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.alloc.rate.norm                avgt   30  1189.332 ±  21.185    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Eden_Space            avgt   30  1540.644 ± 141.078  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Eden_Space.norm       avgt   30  1192.054 ± 111.425    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Old_Gen               avgt   30   152.361 ±  83.066  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Old_Gen.norm          avgt   30   118.345 ±  64.581    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Survivor_Space        avgt   30     0.060 ±   0.118  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.churn.PS_Survivor_Space.norm   avgt   30     0.046 ±   0.091    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.count                          avgt   30   127.000            counts
MetricArchiveMutableAddBenchmark.LogarithmHistogram.addPointToArchiveStream:·gc.time                           avgt   30  1993.000                ms
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline                                                   avgt   30   321.973 ±   2.389   ns/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·cpi                                              avgt          0.485               CPI
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.alloc.rate                                    avgt   30  3046.480 ±  22.064  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.alloc.rate.norm                               avgt   30  1028.508 ±   0.190    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.churn.PS_Eden_Space                           avgt   30  3044.585 ±  51.427  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.churn.PS_Eden_Space.norm                      avgt   30  1027.823 ±  14.260    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.churn.PS_Survivor_Space                       avgt   30     0.074 ±   0.014  MB/sec
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.churn.PS_Survivor_Space.norm                  avgt   30     0.025 ±   0.005    B/op
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.count                                         avgt   30   662.000            counts
MetricArchiveMutableAddBenchmark.LogarithmHistogram.baseline:·gc.time                                          avgt   30   770.000                ms
     */
}
