package ru.yandex.solomon.codec;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Random;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;

import static ru.yandex.solomon.model.point.column.LongValueRandomData.randomLongValue;
import static ru.yandex.solomon.model.point.column.ValueRandomData.randomNum;

/**
 * @author Vladimir Gordiychuk
 */
public final class PointValueDataSets {
    public static final Map<String, Generators> GENERATOR_BY_NAME;

    static {
        Map<String, Generators> data = new LinkedHashMap<>();
        data.put("default", new Generators());
        data.put("1s", gen((index, prev, r) -> 1L));
        data.put("rnd 0,1", gen((index, prev, r) -> r.nextBoolean() ? 1 : 0));
        data.put("rnd 1,2", gen((index, prev, r) -> r.nextBoolean() ? 1 : 2));
        data.put("rnd 2,3", gen((index, prev, r) -> r.nextBoolean() ? 2 : 3));
        data.put("0,1", gen((index, prev, r) -> index % 2));
        data.put("1,2", gen((index, prev, r) -> 1 + index % 2));
        data.put("0..1000", gen((index, prev, r) -> r.nextInt(1000)));
        data.put("1500..1600", gen((index, prev, r) -> r.nextInt(100) + 1500));
        data.put("1000000", gen((index, prev, r) -> r.nextInt(1000000)));
        data.put("seq", gen((index, prev, r) -> prev + 1));
        data.put("seqMultiply", gen((index, prev, r) -> index * index));
        data.put("rnd", new Generators((index, prev, r) -> randomLongValue(r), (index, prev, r) -> randomNum(r)));
        data.put("rndGrow", gen((index, prev, r) -> prev + r.nextLong()));
        GENERATOR_BY_NAME = data;
    }

    private static class Generators {
        LongSupplier longSupplier = (index, prev, r) -> prev;
        DoubleSupplier doubleSupplier = (index, prev, r) -> prev;

        public Generators() {
        }

        public Generators(LongSupplier longSupplier, DoubleSupplier doubleSupplier) {
            this.longSupplier = longSupplier;
            this.doubleSupplier = doubleSupplier;
        }
    }

    public interface PointValueGenerator {
        void fill(int i, AggrPoint point, Random r);
    }

    public static PointValueGenerator gen(MetricType type, String gen) {
        Generators generators = GENERATOR_BY_NAME.get(gen);
        if (generators == null) {
            throw new NullPointerException("Not found generator by name: " + type);
        }

        switch (type) {
            case DGAUGE:
                return (i, point, r) ->
                        point.valueNum = generators.doubleSupplier.supply(i, point.valueNum, r);
            case COUNTER:
            case RATE:
            case IGAUGE:
                return (i, point, r) -> point.longValue = generators.longSupplier.supply(i, point.longValue, r);
            default:
                throw new UnsupportedOperationException("Unsupported type: " + gen);
        }
    }

    private static Generators gen(LongSupplier supplier) {
        DoubleSupplier doubleSupplier = (index, prev, r) -> supplier.supply(index, (long) prev, r);
        return new Generators(supplier, doubleSupplier);
    }

    private interface LongSupplier {
        long supply(int index, long prev, Random r);
    }

    private interface DoubleSupplier {
        double supply(int index, double prev, Random r);
    }
}
