package ru.yandex.solomon.codec.count;

import java.util.function.IntUnaryOperator;
import java.util.stream.IntStream;

import io.netty.util.internal.ThreadLocalRandom;

import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Maksim Leonov
 */
public class CountEncoderBenchmark {

    private static void test(String name, int[] data) {
        long sizeWithoutCount = compress(StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask(), data);
        long sizeWithCount = compress(StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask() | StockpileColumn.COUNT.mask(), data);

        long diff = sizeWithCount - sizeWithoutCount;
        double b = 1.0 * diff / data.length;
        System.out.printf("%-10s: %.1f bits per point\n", name, b);
    }

    private static long compress(int mask, int[] data) {
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, mask);

        AggrPoint point = new AggrPoint();
        point.tsMillis = System.currentTimeMillis();
        point.valueNum = 0;

        for (int count : data) {
            point.count = count;
            stream.writePoint(mask, point);
        }

        return stream.getCompressedData().readableBits();
    }

    private static int[] gen(IntUnaryOperator supplier) {
        return IntStream.range(0, 500).map(supplier::applyAsInt).toArray();
    }

    public static void main(String[] args) {
        ThreadLocalRandom tlr = ThreadUtils.currentThreadLocalRandom();
        test("const", gen(i -> 25));
        test("19+-1", gen(i -> 19 + tlr.nextInt(3) - 1));
        test("107+-10", gen(i -> 107 + tlr.nextInt(21) - 10));
        test("1402+-100", gen(i -> 1402 + tlr.nextInt(101) - 100));
    }

}
