package ru.yandex.solomon.codec.gorillaDouble;

import java.util.Random;
import java.util.function.IntToDoubleFunction;
import java.util.stream.IntStream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Stepan Koltsov
 */
public class GorillaDoubleEncoderBenchmark {
    private static void test(String name, double[] data) {
        final int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, mask);

        AggrPoint point = new AggrPoint();
        point.tsMillis = System.currentTimeMillis();
        for (double value : data) {
            point.valueNum = value;
            stream.writePoint(mask, point);
        }

        long lengthBits = stream.getCompressedData().readableBits();
        double b = 1.0 * lengthBits / data.length;
        double b2 = 1.0 * (lengthBits - 64) / (data.length - 1);
        System.out.printf("%-10s: %.1f %.1f bits per point\n", name, b, b2);
    }

    private static double[] gen(IntToDoubleFunction supplier) {
        return IntStream.range(0, 500).mapToDouble(supplier::applyAsDouble).toArray();
    }

    private static double junk() {
        for (;;) {
            double d = Double.longBitsToDouble(ThreadUtils.currentThreadLocalRandom().nextLong());
            if (Double.isFinite(d)) {
                return d;
            }
        }
    }

    public static void main(String[] args) {
        Random rnd = ThreadUtils.currentThreadLocalRandom();
        test("0s", new double[500]);
        test("1s", Cf.DoubleArray.filled(500, 1.0));
        test("rnd 0,1", gen(i -> rnd.nextBoolean() ? 1 : 0));
        test("rnd 1,2", gen(i -> rnd.nextBoolean() ? 1 : 2));
        test("rnd 2,3", gen(i -> rnd.nextBoolean() ? 2 : 3));
        test("0,1", gen(i -> i % 2));
        test("1,2", gen(i -> 1 + i % 2));
        test("0..1000", gen(i -> rnd.nextInt(1000)));
        test("1500..1600", gen(i -> rnd.nextInt(100) + 1500));
        test("0..1000000", gen(i -> rnd.nextInt(1000000)));
        test("junk", gen(i -> junk()));
        test("grow", gen(i -> i));

        //System.out.println(Arrays.toString(gen));
        //double[] d = new double[] { 0.0, 1.0 };
        //test("xx", d);
    }

}
