package ru.yandex.solomon.codec.histogram;

import java.util.Arrays;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.IntToLongFunction;
import java.util.stream.IntStream;
import java.util.stream.LongStream;

import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class HistogramEncoderBenchmark {
    private static void test(String name, Histogram[] histograms) {
        final int mask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.HIST, mask);

        AggrPoint point = new AggrPoint();
        point.tsMillis = System.currentTimeMillis();
        for (var histogram : histograms) {
            point.histogram = histogram;
            stream.writePoint(mask, point);
        }

        long lengthBits = stream.getCompressedData().readableBits();
        double b = 1.0 * lengthBits / histograms.length;
        double b2 = 1.0 * (lengthBits - 64) / (histograms.length - 1);
        System.out.printf("%-10s: %.1f %.1f bits per point\n", name, b, b2);
    }

    private static Histogram[] gen(double[] bounds, IntToLongFunction supplier) {
        return IntStream.range(0, 500)
                .mapToObj(index -> {
                    long[] buckets = LongStream.range(0, bounds.length)
                            .map(ignore -> supplier.applyAsLong(index))
                            .toArray();

                    return Histogram.newInstance(bounds, buckets);
                })
                .toArray(Histogram[]::new);
    }

    private static long junk() {
        return ThreadLocalRandom.current().nextLong();
    }

    private static Histogram[] zeros(double[] bounds) {
        Histogram[] histograms = new Histogram[500];
        Histogram histogram = Histogram.newInstance(bounds, new long[bounds.length]);
        Arrays.fill(histograms, histogram);
        return histograms;
    }

    private static Histogram[] ones(double[] bounds) {
        Histogram[] histograms = new Histogram[500];
        long[] buckets = new long[bounds.length];
        Arrays.fill(buckets, 1);
        Histogram histogram = Histogram.newInstance(bounds, buckets);
        Arrays.fill(histograms, histogram);
        return histograms;
    }

    public static void main(String[] args) {
        Random rnd = ThreadUtils.currentThreadLocalRandom();
//        double[] bounds = {10, 50, 100, 300, 500, 1_000, 3_000, 5_000, 10_000, 30_000, Long.MAX_VALUE};
        double[] bounds = {10};
        test("0s", zeros(bounds));
        test("1s", ones(bounds));
        test("rnd 0,1", gen(bounds, i -> rnd.nextBoolean() ? 1 : 0));
        test("rnd 1,2", gen(bounds, i -> rnd.nextBoolean() ? 1 : 2));
        test("rnd 2,3", gen(bounds, i -> rnd.nextBoolean() ? 2 : 3));
        test("0,1", gen(bounds, i -> i % 2));
        test("1,2", gen(bounds, i -> 1 + i % 2));
        test("0..1000", gen(bounds, i -> rnd.nextInt(1000)));
        test("1500..1600", gen(bounds, i -> rnd.nextInt(100) + 1500));
        test("0..1000000", gen(bounds, i -> rnd.nextInt(1000000)));
        test("junk", gen(bounds, i -> junk()));
    }
}
